#include "NSGAIIOptimizer.h"
#include <random>
#include <algorithm>
#include <limits>

NSGAIIOptimizer::NSGAIIOptimizer(int dimension, int numObjectives)
    : dimension(dimension), numObjectives(numObjectives), populationSize(100) {
    lowerBounds.resize(dimension, -5.0);
    upperBounds.resize(dimension, 5.0);
}

void NSGAIIOptimizer::setBounds(const std::vector<double>& lower, const std::vector<double>& upper) {
    lowerBounds = lower;
    upperBounds = upper;
}

bool NSGAIIOptimizer::dominates(const Individual& a, const Individual& b) {
    bool atLeastOneBetter = false;
    for (size_t i = 0; i < a.objectives.size(); i++) {
        if (a.objectives[i] > b.objectives[i]) return false;
        if (a.objectives[i] < b.objectives[i]) atLeastOneBetter = true;
    }
    return atLeastOneBetter;
}

void NSGAIIOptimizer::fastNonDominatedSort(std::vector<Individual>& population) {
    for (auto& ind : population) {
        ind.rank = 0;
        ind.crowdingDistance = 0.0;
    }

    std::vector<std::vector<Individual*>> fronts(1);

    for (size_t i = 0; i < population.size(); i++) {
        int dominationCount = 0;
        for (size_t j = 0; j < population.size(); j++) {
            if (i != j && dominates(population[j], population[i])) {
                dominationCount++;
            }
        }
        if (dominationCount == 0) {
            population[i].rank = 0;
            fronts[0].push_back(&population[i]);
        }
    }

    int currentRank = 0;
    while (!fronts[currentRank].empty()) {
        std::vector<Individual*> nextFront;
        for (auto* ind : fronts[currentRank]) {
            for (size_t i = 0; i < population.size(); i++) {
                if (population[i].rank == -1) {
                    bool dominated = false;
                    for (auto* dominator : fronts[currentRank]) {
                        if (dominates(*dominator, population[i])) {
                            dominated = true;
                            break;
                        }
                    }
                    if (!dominated) {
                        population[i].rank = currentRank + 1;
                        nextFront.push_back(&population[i]);
                    }
                }
            }
        }
        if (!nextFront.empty()) {
            fronts.push_back(nextFront);
            currentRank++;
        } else {
            break;
        }
    }
}

void NSGAIIOptimizer::calculateCrowdingDistance(std::vector<Individual>& front) {
    if (front.empty()) return;

    int n = front.size();
    int m = numObjectives;

    for (auto& ind : front) {
        ind.crowdingDistance = 0.0;
    }

    for (int obj = 0; obj < m; obj++) {
        std::sort(front.begin(), front.end(),
                  [obj](const Individual& a, const Individual& b) {
                      return a.objectives[obj] < b.objectives[obj];
                  });

        front[0].crowdingDistance = std::numeric_limits<double>::infinity();
        front[n - 1].crowdingDistance = std::numeric_limits<double>::infinity();

        double range = front[n - 1].objectives[obj] - front[0].objectives[obj];
        if (range > 0) {
            for (int i = 1; i < n - 1; i++) {
                front[i].crowdingDistance +=
                    (front[i + 1].objectives[obj] - front[i - 1].objectives[obj]) / range;
            }
        }
    }
}

std::vector<NSGAIIOptimizer::Individual> NSGAIIOptimizer::optimize(ObjectiveFunction func, int maxGenerations) {
    std::random_device rd;
    std::mt19937 gen(rd());

    std::vector<Individual> population(populationSize);

    // Initialize population
    for (int i = 0; i < populationSize; i++) {
        population[i].variables.resize(dimension);
        for (int j = 0; j < dimension; j++) {
            std::uniform_real_distribution<double> dist(lowerBounds[j], upperBounds[j]);
            population[i].variables[j] = dist(gen);
        }
        population[i].objectives = func(population[i].variables);
    }

    // Evolution loop
    for (int generation = 0; generation < maxGenerations; generation++) {
        std::vector<Individual> offspring;

        // Selection and crossover (simplified)
        for (int i = 0; i < populationSize / 2; i++) {
            std::uniform_int_distribution<int> dist(0, populationSize - 1);
            int p1 = dist(gen);
            int p2 = dist(gen);

            Individual child;
            child.variables.resize(dimension);
            for (int j = 0; j < dimension; j++) {
                child.variables[j] = (population[p1].variables[j] + population[p2].variables[j]) / 2.0;
            }
            child.objectives = func(child.variables);
            offspring.push_back(child);
        }

        population.insert(population.end(), offspring.begin(), offspring.end());

        fastNonDominatedSort(population);

        std::sort(population.begin(), population.end(),
                  [](const Individual& a, const Individual& b) {
                      if (a.rank != b.rank) return a.rank < b.rank;
                      return a.crowdingDistance > b.crowdingDistance;
                  });

        population.resize(populationSize);
    }

    return population;
}
