#pragma once
#include "Mesh.h"
#include "../sdf/SDFEngine.h"
#include <cmath>

class MeshGenerator {
public:
    // SDF to mesh conversion methods
    static Mesh marchingCubes(const SDFEngine& sdf, int resolution = 32);
    static Mesh surfels(const SDFEngine& sdf, int resolution = 32);

    // Parametric surfaces
    static Mesh createSphere(float radius, int segments = 32, int rings = 16);
    static Mesh createCube(float size);
    static Mesh createCylinder(float radius, float height, int segments = 32);
    static Mesh createTorus(float majorRadius, float minorRadius, int segments = 32, int sides = 16);
    static Mesh createCone(float radius, float height, int segments = 32);
    static Mesh createPlane(float width, float height, int segmentsX = 10, int segmentsY = 10);

    // Platonic solids
    static Mesh createTetrahedron(float size);
    static Mesh createOctahedron(float size);
    static Mesh createDodecahedron(float size);
    static Mesh createIcosahedron(float size);

    // Classic models
    static Mesh createTeapot(float scale = 1.0f, int subdivisions = 8);
    static Mesh createUtahTeapot(float scale = 1.0f, int subdivisions = 8);
    static Mesh createOilTank(float radius = 1.0f, float height = 2.0f, int segments = 32);

    // New primitives
    static Mesh createGeoSphere(float radius, int subdivisions);
    static Mesh createTube(float radius, float height, float thickness, int segments);
    static Mesh createPyramid(float baseSize, float height, int subdivisions);
    static Mesh createHedra(float radius, int sides, int subdivision);
    static Mesh createTorusKnot(int p, int q, float tubeRadius, int segments);
    static Mesh createCapsule(float radius, float height, int segments);
    static Mesh createSpindle(float radius, float height, int segments);
    static Mesh createLExt(float width, float height, float thickness, float depth);
    static Mesh createCExt(float width, float height, float thickness, float depth);
    static Mesh createGengon(float radius, int sides, float fillet, float height);
    static Mesh createRingWave(float radius, float thickness, float waveAmplitude, int waveFrequency, int segments);
    static Mesh createHose(float radius, float length, float bendAngle, int segments);
    static Mesh createPrism(float radius, int sides, float height, int heightSegments);

    // Advanced shapes
    static Mesh createMobiusStrip(float radius, float width, int segments = 64);
    static Mesh createKleinBottle(int segments = 32);
    static Mesh createTrefoilKnot(float scale, int segments = 64);

    // Grid and lattice
    static Mesh createGrid(int width, int height, float spacing);
    static Mesh createHexGrid(int rings, float radius);

    // Utility functions
    static void subdivide(Mesh& mesh, int iterations = 1);
    static void extrude(Mesh& mesh, const Vec3& direction, float distance);
    static Mesh lathe(const std::vector<Vec3>& profile, int segments);
    static Mesh loft(const std::vector<std::vector<Vec3>>& sections);

private:
    static int edgeTable[256];
    static int triTable[256][16];
    static Vec3 vertexInterp(float isolevel, const Vec3& p1, const Vec3& p2, float v1, float v2);
};
