/**
 * @file Mesh.h
 * @brief Core mesh data structures and operations
 *
 * This header defines the fundamental mesh representation used throughout
 * the application. The Mesh class provides a half-edge-free, index-based
 * triangle mesh structure with support for normals, colors, and texture coordinates.
 *
 * Key features:
 * - Lightweight vertex and triangle structures
 * - Automatic normal computation (smooth shading)
 * - Mesh transformation (translate, rotate, scale)
 * - Mesh merging/combination operations
 * - Vertex colors and texture coordinate support
 *
 * Copyright © 2025 Linus Suter
 * Released under the GNU/GPL License
 */

#pragma once

#include "../math/Vec3.h"
#include <vector>
#include <string>

/**
 * @brief Vertex structure containing position, normal, color, and UV data
 *
 * Represents a single vertex in a mesh with all associated attributes.
 * The default constructor creates a white vertex at the origin.
 */
struct Vertex {
    Vec3 position;  ///< 3D position in model space
    Vec3 normal;    ///< Surface normal (unit vector)
    Vec3 color;     ///< RGB color (0-1 range), default white
    float u, v;     ///< Texture coordinates (0-1 range)

    /**
     * @brief Default constructor - Creates vertex at origin with white color
     */
    Vertex() : position(), normal(), color(1, 1, 1), u(0), v(0) {}

    /**
     * @brief Position-only constructor - Creates vertex with specified position
     * @param p Vertex position
     */
    Vertex(const Vec3& p) : position(p), normal(), color(1, 1, 1), u(0), v(0) {}

    /**
     * @brief Position and normal constructor
     * @param p Vertex position
     * @param n Vertex normal
     */
    Vertex(const Vec3& p, const Vec3& n) : position(p), normal(n), color(1, 1, 1), u(0), v(0) {}
};

/**
 * @brief Triangle structure using vertex indices
 *
 * Represents a triangular face by storing indices into the vertex array.
 * Vertices are stored in counter-clockwise order (right-hand rule).
 */
struct Triangle {
    unsigned int v0, v1, v2;  ///< Vertex indices (CCW winding order)

    /**
     * @brief Constructor - Creates triangle from three vertex indices
     * @param a First vertex index
     * @param b Second vertex index
     * @param c Third vertex index
     */
    Triangle(unsigned int a, unsigned int b, unsigned int c) : v0(a), v1(b), v2(c) {}
};

/**
 * @brief Main mesh class for storing and manipulating triangle meshes
 *
 * Provides a simple, efficient triangle mesh representation using indexed
 * geometry. Supports common operations like normal computation, transformations,
 * and mesh merging.
 */
class Mesh {
private:
    std::vector<Vertex> vertices;      ///< Vertex array
    std::vector<Triangle> triangles;   ///< Triangle index array
    std::string name;                  ///< Mesh name for identification

public:
    /**
     * @brief Default constructor - Creates an empty mesh
     */
    Mesh() : name("Mesh") {}
    void getBounds(Vec3& min, Vec3& max) const;
    /**
     * @brief Clears all vertices and triangles from the mesh
     */
    void clear();

    // ===== Vertex Operations =====
    /**
     * @brief Adds a vertex to the mesh
     * @param v Vertex to add
     */
    void addVertex(const Vertex& v) {
        vertices.push_back(v);
    }

    /**
     * @brief Gets the number of vertices in the mesh
     * @return Vertex count
     */
    size_t getVertexCount() const { return vertices.size(); }

    /**
     * @brief Gets a reference to a vertex by index
     * @param i Vertex index
     * @return Reference to vertex at index i
     */
    Vertex& getVertex(size_t i) { return vertices[i]; }

    /**
     * @brief Gets a const reference to a vertex by index
     * @param i Vertex index
     * @return Const reference to vertex at index i
     */
    const Vertex& getVertex(size_t i) const { return vertices[i]; }

    /**
     * @brief Gets the entire vertex array
     * @return Reference to vertex vector
     */
    std::vector<Vertex>& getVertices() { return vertices; }

    /**
     * @brief Gets the entire vertex array (const)
     * @return Const reference to vertex vector
     */
    const std::vector<Vertex>& getVertices() const { return vertices; }

    // ===== Triangle Operations =====

    /**
     * @brief Adds a triangle to the mesh using vertex indices
     * @param v0 First vertex index
     * @param v1 Second vertex index
     * @param v2 Third vertex index
     */
    void addTriangle(unsigned int v0, unsigned int v1, unsigned int v2) {
        triangles.push_back(Triangle(v0, v1, v2));
    }

    /**
     * @brief Adds a triangle to the mesh
     * @param t Triangle to add
     */
    void addTriangle(const Triangle& t) {
        triangles.push_back(t);
    }

    /**
     * @brief Gets the number of triangles in the mesh
     * @return Triangle count
     */
    size_t getTriangleCount() const { return triangles.size(); }

    /**
     * @brief Gets a reference to a triangle by index
     * @param i Triangle index
     * @return Reference to triangle at index i
     */
    Triangle& getTriangle(size_t i) { return triangles[i]; }

    /**
     * @brief Gets a const reference to a triangle by index
     * @param i Triangle index
     * @return Const reference to triangle at index i
     */
    const Triangle& getTriangle(size_t i) const { return triangles[i]; }

    /**
     * @brief Gets the entire triangle array
     * @return Reference to triangle vector
     */
    std::vector<Triangle>& getTriangles() { return triangles; }

    /**
     * @brief Gets the entire triangle array (const)
     * @return Const reference to triangle vector
     */
    const std::vector<Triangle>& getTriangles() const { return triangles; }

    // ===== Mesh Properties =====

    /**
     * @brief Sets the mesh name
     * @param n New mesh name
     */
    void setName(const std::string& n) { name = n; }

    /**
     * @brief Gets the mesh name
     * @return Current mesh name
     */
    const std::string& getName() const { return name; }

    // ===== Mesh Operations =====

    /**
     * @brief Computes smooth vertex normals from face normals
     *
     * Uses area-weighted averaging of adjacent face normals to compute
     * smooth per-vertex normals for lighting calculations. This provides
     * smooth shading across the mesh surface.
     *
     * Algorithm:
     * 1. Reset all vertex normals to zero
     * 2. For each triangle, compute face normal and add to vertices
     * 3. Normalize all vertex normals
     */
    void computeNormals() {
        // Reset all vertex normals to zero
        for (auto& v : vertices) {
            v.normal = Vec3(0, 0, 0);
        }

        // Accumulate face normals to vertices (area-weighted)
        for (const auto& tri : triangles) {
            const Vec3& p0 = vertices[tri.v0].position;
            const Vec3& p1 = vertices[tri.v1].position;
            const Vec3& p2 = vertices[tri.v2].position;

            // Compute face normal using cross product
            Vec3 normal = (p1 - p0).cross(p2 - p0);

            // Add face normal to each vertex (weighted by triangle area)
            vertices[tri.v0].normal += normal;
            vertices[tri.v1].normal += normal;
            vertices[tri.v2].normal += normal;
        }

        // Normalize all vertex normals to unit length
        for (auto& v : vertices) {
            v.normal.normalize();
        }
    }

    /**
     * @brief Applies transformation to all mesh vertices
     * @param translation Translation offset to apply
     * @param scale Scale factors for X, Y, Z axes
     * @param rotation Euler rotation angles in radians (X, Y, Z order)
     *
     * Transforms all vertices by:
     * 1. Scaling
     * 2. Rotating (Euler angles: Z, Y, X order)
     * 3. Translating
     * 4. Recomputing normals
     */
    void transform(const Vec3& translation, const Vec3& scale, const Vec3& rotation) {
        for (auto& v : vertices) {
            // Step 1: Apply scale
            v.position.x *= scale.x;
            v.position.y *= scale.y;
            v.position.z *= scale.z;

            // Step 2: Apply Euler rotations (Z -> Y -> X order)
            float cx = std::cos(rotation.x), sx = std::sin(rotation.x);
            float cy = std::cos(rotation.y), sy = std::sin(rotation.y);
            float cz = std::cos(rotation.z), sz = std::sin(rotation.z);

            float x = v.position.x;
            float y = v.position.y;
            float z = v.position.z;

            // Rotate around Z-axis
            float x1 = x * cz - y * sz;
            float y1 = x * sz + y * cz;

            // Rotate around Y-axis
            float x2 = x1 * cy + z * sy;
            float z1 = -x1 * sy + z * cy;

            // Rotate around X-axis
            float y2 = y1 * cx - z1 * sx;
            float z2 = y1 * sx + z1 * cx;

            v.position = Vec3(x2, y2, z2);

            // Step 3: Apply translation
            v.position += translation;
        }

        // Recompute normals after transformation
        computeNormals();
    }

    /**
     * @brief Merges another mesh into this mesh
     * @param other The mesh to merge into this one
     *
     * Combines the geometry of two meshes by appending vertices and
     * triangles. Triangle indices are adjusted to reference the correct
     * vertices in the combined vertex array.
     */
    void merge(const Mesh& other) {
        // Store current vertex count for index offset
        size_t vertexOffset = vertices.size();

        // Append all vertices from other mesh
        vertices.insert(vertices.end(), other.vertices.begin(), other.vertices.end());

        // Append triangles with adjusted indices
        for (const auto& tri : other.triangles) {
            triangles.push_back(Triangle(
                tri.v0 + vertexOffset,
                tri.v1 + vertexOffset,
                tri.v2 + vertexOffset
            ));
        }
    }
};
