#include "FBXExporter.h"
#include <fstream>
#include <sstream>
#include <ctime>
#include <iomanip>

void FBXExporter::writeHeader(std::ofstream& file) {
    // FBX header with proper format
    file << "; FBX 7.4.0 project file\n";
    file << "; Created by codewelt.com\n";
    file << "; ----------------------------------------------------\n\n";

    // FBX header extension section
    file << "FBXHeaderExtension:  {\n";
    file << "\tFBXHeaderVersion: 1003\n";
    file << "\tFBXVersion: 7400\n";
    file << "\tCreationTimeStamp:  {\n";
    file << "\t\tVersion: 1000\n";

    // Get current time
    std::time_t now = std::time(nullptr);
    std::tm* timeinfo = std::localtime(&now);

    file << "\t\tYear: " << (1900 + timeinfo->tm_year) << "\n";
    file << "\t\tMonth: " << (1 + timeinfo->tm_mon) << "\n";
    file << "\t\tDay: " << timeinfo->tm_mday << "\n";
    file << "\t\tHour: " << timeinfo->tm_hour << "\n";
    file << "\t\tMinute: " << timeinfo->tm_min << "\n";
    file << "\t\tSecond: " << timeinfo->tm_sec << "\n";
    file << "\t\tMillisecond: 0\n";
    file << "\t}\n";
    file << "\tCreator: \"codewelt.com/mesher\"\n";
    file << "}\n\n";

    // Global settings
    file << "GlobalSettings:  {\n";
    file << "\tVersion: 1000\n";
    file << "\tProperties70:  {\n";
    file << "\t\tP: \"UpAxis\", \"int\", \"Integer\", \"\",1\n";
    file << "\t\tP: \"UpAxisSign\", \"int\", \"Integer\", \"\",1\n";
    file << "\t\tP: \"FrontAxis\", \"int\", \"Integer\", \"\",2\n";
    file << "\t\tP: \"FrontAxisSign\", \"int\", \"Integer\", \"\",1\n";
    file << "\t\tP: \"CoordAxis\", \"int\", \"Integer\", \"\",0\n";
    file << "\t\tP: \"CoordAxisSign\", \"int\", \"Integer\", \"\",1\n";
    file << "\t\tP: \"OriginalUpAxis\", \"int\", \"Integer\", \"\",1\n";
    file << "\t\tP: \"OriginalUpAxisSign\", \"int\", \"Integer\", \"\",1\n";
    file << "\t\tP: \"UnitScaleFactor\", \"double\", \"Number\", \"\",1\n";
    file << "\t\tP: \"OriginalUnitScaleFactor\", \"double\", \"Number\", \"\",1\n";
    file << "\t}\n";
    file << "}\n\n";

    // Documents section
    file << "Documents:  {\n";
    file << "\tCount: 1\n";
    file << "\tDocument: 1234567890, \"\", \"Scene\" {\n";
    file << "\t\tProperties70:  {\n";
    file << "\t\t\tP: \"SourceObject\", \"object\", \"\", \"\"\n";
    file << "\t\t\tP: \"ActiveAnimStackName\", \"KString\", \"\", \"\", \"\"\n";
    file << "\t\t}\n";
    file << "\t\tRootNode: 0\n";
    file << "\t}\n";
    file << "}\n\n";

    // References section
    file << "References:  {\n";
    file << "}\n\n";

    // Definitions section
    file << "Definitions:  {\n";
    file << "\tVersion: 100\n";
    file << "\tCount: 3\n";

    // Model definition
    file << "\tObjectType: \"Model\" {\n";
    file << "\t\tCount: 1\n";
    file << "\t}\n";

    // Geometry definition
    file << "\tObjectType: \"Geometry\" {\n";
    file << "\t\tCount: 1\n";
    file << "\t}\n";

    // Material definition
    file << "\tObjectType: \"Material\" {\n";
    file << "\t\tCount: 1\n";
    file << "\t}\n";

    file << "}\n\n";

    // Objects section header
    file << "Objects:  {\n";
}

void FBXExporter::writeMeshData(std::ofstream& file, const Mesh& mesh) {
    // Model node
    file << "\tModel: 2345678901, \"Model::" << mesh.getName() << "\", \"Mesh\" {\n";
    file << "\t\tVersion: 232\n";
    file << "\t\tProperties70:  {\n";
    file << "\t\t\tP: \"RotationActive\", \"bool\", \"\", \"\",1\n";
    file << "\t\t\tP: \"InheritType\", \"enum\", \"\", \"\",1\n";
    file << "\t\t\tP: \"ScalingMax\", \"Vector3D\", \"Vector\", \"\",0,0,0\n";
    file << "\t\t\tP: \"DefaultAttributeIndex\", \"int\", \"Integer\", \"\",0\n";
    file << "\t\t}\n";
    file << "\t\tShading: T\n";
    file << "\t\tCulling: \"CullingOff\"\n";
    file << "\t}\n";

    // Geometry
    file << "\tGeometry: 3456789012, \"Geometry::\", \"Mesh\" {\n";

    // Vertices
    file << "\t\tVertices: *" << (mesh.getVertexCount() * 3) << " {\n\t\t\ta: ";
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& p = mesh.getVertex(i).position;
        file << p.x << "," << p.y << "," << p.z;
        if (i < mesh.getVertexCount() - 1) file << ",";
    }
    file << "\n\t\t}\n";

    // Polygon vertex indices (negative values mark end of polygon)
    file << "\t\tPolygonVertexIndex: *" << (mesh.getTriangleCount() * 3) << " {\n\t\t\ta: ";
    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        // Last index of each triangle is negated and subtracted by 1
        file << tri.v0 << "," << tri.v1 << "," << (-(int)tri.v2 - 1);
        if (i < mesh.getTriangleCount() - 1) file << ",";
    }
    file << "\n\t\t}\n";

    // Edges (optional but helps with import)
    file << "\t\tGeometryVersion: 124\n";

    // Layer element normal
    file << "\t\tLayerElementNormal: 0 {\n";
    file << "\t\t\tVersion: 101\n";
    file << "\t\t\tName: \"\"\n";
    file << "\t\t\tMappingInformationType: \"ByVertice\"\n";
    file << "\t\t\tReferenceInformationType: \"Direct\"\n";
    file << "\t\t\tNormals: *" << (mesh.getVertexCount() * 3) << " {\n\t\t\t\ta: ";
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& n = mesh.getVertex(i).normal;
        file << n.x << "," << n.y << "," << n.z;
        if (i < mesh.getVertexCount() - 1) file << ",";
    }
    file << "\n\t\t\t}\n";
    file << "\t\t}\n";

    // Layer element material
    file << "\t\tLayerElementMaterial: 0 {\n";
    file << "\t\t\tVersion: 101\n";
    file << "\t\t\tName: \"\"\n";
    file << "\t\t\tMappingInformationType: \"AllSame\"\n";
    file << "\t\t\tReferenceInformationType: \"IndexToDirect\"\n";
    file << "\t\t\tMaterials: *1 {\n\t\t\t\ta: 0\n\t\t\t}\n";
    file << "\t\t}\n";

    // Layer
    file << "\t\tLayer: 0 {\n";
    file << "\t\t\tVersion: 100\n";
    file << "\t\t\tLayerElement:  {\n";
    file << "\t\t\t\tType: \"LayerElementNormal\"\n";
    file << "\t\t\t\tTypedIndex: 0\n";
    file << "\t\t\t}\n";
    file << "\t\t\tLayerElement:  {\n";
    file << "\t\t\t\tType: \"LayerElementMaterial\"\n";
    file << "\t\t\t\tTypedIndex: 0\n";
    file << "\t\t\t}\n";
    file << "\t\t}\n";

    file << "\t}\n";

    // Material
    file << "\tMaterial: 4567890123, \"Material::DefaultMaterial\", \"\" {\n";
    file << "\t\tVersion: 102\n";
    file << "\t\tShadingModel: \"phong\"\n";
    file << "\t\tMultiLayer: 0\n";
    file << "\t\tProperties70:  {\n";
    file << "\t\t\tP: \"Diffuse\", \"Color\", \"\", \"A\",0.8,0.8,0.8\n";
    file << "\t\t\tP: \"DiffuseColor\", \"Color\", \"\", \"A\",0.8,0.8,0.8\n";
    file << "\t\t\tP: \"Emissive\", \"Color\", \"\", \"A\",0,0,0\n";
    file << "\t\t\tP: \"Ambient\", \"Color\", \"\", \"A\",0.2,0.2,0.2\n";
    file << "\t\t\tP: \"Specular\", \"Color\", \"\", \"A\",0.2,0.2,0.2\n";
    file << "\t\t\tP: \"Shininess\", \"double\", \"Number\", \"\",20\n";
    file << "\t\t\tP: \"Opacity\", \"double\", \"Number\", \"\",1.0\n";
    file << "\t\t\tP: \"Reflectivity\", \"double\", \"Number\", \"\",0\n";
    file << "\t\t}\n";
    file << "\t}\n";
}

void FBXExporter::writeAnimationData(std::ofstream& file, const VertexAnimator& animator) {
    file << "\tAnimationStack: 5678901234, \"AnimStack::Take 001\", \"\" {\n";
    file << "\t\tProperties70:  {\n";
    file << "\t\t\tP: \"LocalStart\", \"KTime\", \"Time\", \"\",0\n";
    file << "\t\t\tP: \"LocalStop\", \"KTime\", \"Time\", \"\","
         << (long long)(animator.getDuration() * 46186158000) << "\n";
    file << "\t\t\tP: \"ReferenceStart\", \"KTime\", \"Time\", \"\",0\n";
    file << "\t\t\tP: \"ReferenceStop\", \"KTime\", \"Time\", \"\","
         << (long long)(animator.getDuration() * 46186158000) << "\n";
    file << "\t\t}\n";
    file << "\t}\n";
}

bool FBXExporter::exportMesh(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) {
        return false;
    }

    // Set precision for floating point numbers
    file << std::fixed << std::setprecision(6);

    writeHeader(file);
    writeMeshData(file, mesh);

    // Close Objects section
    file << "}\n\n";

    // Connections section - links objects together
    file << "Connections:  {\n";
    file << "\t;Model::Mesh, Model::RootNode\n";
    file << "\tC: \"OO\",2345678901,0\n";
    file << "\t\n";
    file << "\t;Geometry::, Model::Mesh\n";
    file << "\tC: \"OO\",3456789012,2345678901\n";
    file << "\t\n";
    file << "\t;Material::DefaultMaterial, Model::Mesh\n";
    file << "\tC: \"OO\",4567890123,2345678901\n";
    file << "}\n";

    file.close();
    return true;
}

bool FBXExporter::exportAnimatedMesh(const std::string& filepath,
                                     const Mesh& mesh,
                                     const VertexAnimator& animator) {
    std::ofstream file(filepath);
    if (!file.is_open()) {
        return false;
    }

    // Set precision for floating point numbers
    file << std::fixed << std::setprecision(6);

    writeHeader(file);
    writeMeshData(file, mesh);
    writeAnimationData(file, animator);

    // Close Objects section
    file << "}\n\n";

    // Connections section - links objects together
    file << "Connections:  {\n";
    file << "\t;Model::Mesh, Model::RootNode\n";
    file << "\tC: \"OO\",2345678901,0\n";
    file << "\t\n";
    file << "\t;Geometry::, Model::Mesh\n";
    file << "\tC: \"OO\",3456789012,2345678901\n";
    file << "\t\n";
    file << "\t;Material::DefaultMaterial, Model::Mesh\n";
    file << "\tC: \"OO\",4567890123,2345678901\n";
    file << "\t\n";
    file << "\t;AnimationStack::Take 001, Model::Mesh\n";
    file << "\tC: \"OO\",5678901234,2345678901\n";
    file << "}\n";

    file.close();
    return true;
}
