#pragma once
#include "../mesh/Mesh.h"
#include "../math/Vec3.h"
#include <vector>

class PointCloudMeshing {
public:
    enum MeshingMethod {
        POINTS_ONLY = 0,           // Just render as points
        CONVEX_HULL = 1,           // 3D convex hull
        DELAUNAY_3D = 2,           // 3D Delaunay triangulation
        ALPHA_SHAPES = 3,          // Alpha shapes (generalized convex hull)
        BALL_PIVOTING = 4,         // Ball pivoting algorithm
        POWER_CRUST = 5,           // Power crust reconstruction
        POISSON_RECONSTRUCTION = 6, // Poisson surface reconstruction
        MARCHING_CUBES_DENSITY = 7, // Marching cubes on density field
        SPLATS = 8,                // Point splats/disks
        METABALLS = 9              // Metaball/blobby surface
    };

    // Convert point cloud to mesh using specified method
    static Mesh pointsToMesh(const std::vector<Vec3>& points, MeshingMethod method, float parameter = 1.0f);

private:
    // Individual meshing algorithms
    static Mesh convexHull(const std::vector<Vec3>& points);
    static Mesh delaunay3D(const std::vector<Vec3>& points);
    static Mesh alphaShapes(const std::vector<Vec3>& points, float alpha);
    static Mesh ballPivoting(const std::vector<Vec3>& points, float radius);
    static Mesh powerCrust(const std::vector<Vec3>& points);
    static Mesh poissonReconstruction(const std::vector<Vec3>& points, int depth);
    static Mesh marchingCubesDensity(const std::vector<Vec3>& points, float voxelSize);
    static Mesh splats(const std::vector<Vec3>& points, float splatSize);
    static Mesh metaballs(const std::vector<Vec3>& points, float radius);
};
