#include "PointCloudMeshing.h"
#include "../mesh/MeshGenerator.h"
#include "../sdf/SDFEngine.h"
#include <algorithm>
#include <map>
#include <set>
#include <cmath>

Mesh PointCloudMeshing::pointsToMesh(const std::vector<Vec3>& points, MeshingMethod method, float parameter) {
    if (points.empty()) return Mesh();

    // Limit point count to prevent crashes
    const size_t maxPoints = 2000;
    std::vector<Vec3> limitedPoints;

    if (points.size() > maxPoints) {
        // Downsample by taking every Nth point
        size_t step = points.size() / maxPoints;
        for (size_t i = 0; i < points.size(); i += step) {
            limitedPoints.push_back(points[i]);
            if (limitedPoints.size() >= maxPoints) break;
        }
    } else {
        limitedPoints = points;
    }

    switch (method) {
        case POINTS_ONLY:
            {
                Mesh mesh;
                for (const Vec3& p : limitedPoints) {
                    mesh.addVertex(Vertex(p, Vec3(0, 1, 0)));
                }
                return mesh;
            }
        case CONVEX_HULL:
            return convexHull(limitedPoints);
        case DELAUNAY_3D:
            return delaunay3D(limitedPoints);
        case ALPHA_SHAPES:
            return alphaShapes(limitedPoints, parameter);
        case BALL_PIVOTING:
            return ballPivoting(limitedPoints, parameter);
        case POWER_CRUST:
            return powerCrust(limitedPoints);
        case POISSON_RECONSTRUCTION:
            return poissonReconstruction(limitedPoints, (int)parameter);
        case MARCHING_CUBES_DENSITY:
            return marchingCubesDensity(limitedPoints, parameter);
        case SPLATS:
            return splats(limitedPoints, parameter);
        case METABALLS:
            return metaballs(limitedPoints, parameter);
        default:
            return Mesh();
    }
}

Mesh PointCloudMeshing::convexHull(const std::vector<Vec3>& points) {
    if (points.size() < 4) return Mesh();

    Mesh mesh;

    // Add all points as vertices
    for (const Vec3& p : points) {
        mesh.addVertex(Vertex(p, Vec3(0, 1, 0)));
    }

    // Find bounding points
    size_t minX = 0, maxX = 0, minY = 0, maxY = 0, minZ = 0, maxZ = 0;
    for (size_t i = 0; i < points.size(); i++) {
        if (points[i].x < points[minX].x) minX = i;
        if (points[i].x > points[maxX].x) maxX = i;
        if (points[i].y < points[minY].y) minY = i;
        if (points[i].y > points[maxY].y) maxY = i;
        if (points[i].z < points[minZ].z) minZ = i;
        if (points[i].z > points[maxZ].z) maxZ = i;
    }

    // Create triangles connecting extremal points
    if (minX != maxX && minY != maxY && minZ != maxZ) {
        mesh.addTriangle(minX, minY, minZ);
        mesh.addTriangle(maxX, maxY, maxZ);
        mesh.addTriangle(minX, maxY, minZ);
        mesh.addTriangle(maxX, minY, maxZ);
        mesh.addTriangle(minX, minY, maxZ);
        mesh.addTriangle(maxX, maxY, minZ);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh PointCloudMeshing::delaunay3D(const std::vector<Vec3>& points) {
    // Fast approximation - only use subset of points
    Mesh mesh;

    size_t maxPoints = std::min(points.size(), size_t(50));
    for (size_t i = 0; i < maxPoints; i++) {
        mesh.addVertex(Vertex(points[i], Vec3(0, 1, 0)));
    }

    // Simple triangulation connecting nearby points (limited iterations)
    for (size_t i = 0; i < maxPoints && i < 30; i++) {
        for (size_t j = i + 1; j < maxPoints && j < 30; j++) {
            for (size_t k = j + 1; k < maxPoints && k < 30; k++) {
                float d1 = (points[i] - points[j]).length();
                float d2 = (points[j] - points[k]).length();
                float d3 = (points[k] - points[i]).length();

                if (d1 < 0.5f && d2 < 0.5f && d3 < 0.5f) {
                    mesh.addTriangle(i, j, k);
                }
            }
        }
    }

    if (mesh.getTriangleCount() == 0) {
        // Fallback to simple approach
        return convexHull(points);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh PointCloudMeshing::alphaShapes(const std::vector<Vec3>& points, float alpha) {
    // Use limited points for performance
    Mesh mesh;

    size_t maxPoints = std::min(points.size(), size_t(50));
    for (size_t i = 0; i < maxPoints; i++) {
        mesh.addVertex(Vertex(points[i], Vec3(0, 1, 0)));
    }

    // Connect points within alpha radius (limited iterations)
    int triangleCount = 0;
    const int maxTriangles = 200;

    for (size_t i = 0; i < maxPoints && triangleCount < maxTriangles; i++) {
        for (size_t j = i + 1; j < maxPoints && triangleCount < maxTriangles; j++) {
            for (size_t k = j + 1; k < maxPoints && triangleCount < maxTriangles; k++) {
                Vec3 center = (points[i] + points[j] + points[k]) * (1.0f / 3.0f);
                float r1 = (points[i] - center).length();
                float r2 = (points[j] - center).length();
                float r3 = (points[k] - center).length();

                if (r1 < alpha && r2 < alpha && r3 < alpha) {
                    mesh.addTriangle(i, j, k);
                    triangleCount++;
                }
            }
        }
    }

    if (triangleCount == 0) {
        return convexHull(points);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh PointCloudMeshing::ballPivoting(const std::vector<Vec3>& points, float radius) {
    // Simplified and fast version
    return alphaShapes(points, radius);
}

Mesh PointCloudMeshing::powerCrust(const std::vector<Vec3>& points) {
    // Simplified - use convex hull
    return convexHull(points);
}

Mesh PointCloudMeshing::poissonReconstruction(const std::vector<Vec3>& points, int depth) {
    // Use marching cubes density instead
    return marchingCubesDensity(points, 0.2f);
}

Mesh PointCloudMeshing::marchingCubesDensity(const std::vector<Vec3>& points, float voxelSize) {
    if (points.empty()) return Mesh();

    // Limit points for performance
    size_t maxPoints = std::min(points.size(), size_t(500));

    // Find bounding box
    Vec3 minBound = points[0];
    Vec3 maxBound = points[0];
    for (size_t i = 0; i < maxPoints; i++) {
        minBound.x = std::min(minBound.x, points[i].x);
        minBound.y = std::min(minBound.y, points[i].y);
        minBound.z = std::min(minBound.z, points[i].z);
        maxBound.x = std::max(maxBound.x, points[i].x);
        maxBound.y = std::max(maxBound.y, points[i].y);
        maxBound.z = std::max(maxBound.z, points[i].z);
    }

    // Create density field SDF
    SDFEngine sdf;
    float kernelRadius = voxelSize * 3.0f;

    sdf.setSDF([&points, kernelRadius, maxPoints](const Vec3& p) {
        float density = 0.0f;
        for (size_t i = 0; i < maxPoints; i++) {
            float dist = (p - points[i]).length();
            if (dist < kernelRadius) {
                // Gaussian-like kernel
                density += std::exp(-(dist * dist) / (kernelRadius * kernelRadius));
            }
        }
        return 0.5f - density; // Isosurface at 0
    });

    sdf.setBounds(minBound + Vec3(-kernelRadius, -kernelRadius, -kernelRadius),
                  maxBound + Vec3(kernelRadius, kernelRadius, kernelRadius));

    // Use lower resolution for speed
    int resolution = 24;

    return MeshGenerator::marchingCubes(sdf, resolution);
}

Mesh PointCloudMeshing::splats(const std::vector<Vec3>& points, float splatSize) {
    Mesh mesh;

    // Limit points for performance
    size_t maxPoints = std::min(points.size(), size_t(200));

    // Create disk/splat for each point
    const int segments = 6; // Reduced from 8 for performance
    for (size_t idx = 0; idx < maxPoints; idx++) {
        const Vec3& center = points[idx];
        size_t baseIdx = mesh.getVertexCount();

        // Create disk vertices
        mesh.addVertex(Vertex(center, Vec3(0, 0, 1)));
        for (int i = 0; i < segments; i++) {
            float angle = (i / (float)segments) * 2.0f * M_PI;
            Vec3 offset(std::cos(angle) * splatSize, std::sin(angle) * splatSize, 0);
            mesh.addVertex(Vertex(center + offset, Vec3(0, 0, 1)));
        }

        // Create disk triangles
        for (int i = 0; i < segments; i++) {
            int next = (i + 1) % segments;
            mesh.addTriangle(baseIdx, baseIdx + 1 + i, baseIdx + 1 + next);
        }
    }

    return mesh;
}

Mesh PointCloudMeshing::metaballs(const std::vector<Vec3>& points, float radius) {
    if (points.empty()) return Mesh();

    // Limit points for performance
    size_t maxPoints = std::min(points.size(), size_t(50));

    // Find bounding box
    Vec3 minBound = points[0];
    Vec3 maxBound = points[0];
    for (size_t i = 0; i < maxPoints; i++) {
        minBound.x = std::min(minBound.x, points[i].x);
        minBound.y = std::min(minBound.y, points[i].y);
        minBound.z = std::min(minBound.z, points[i].z);
        maxBound.x = std::max(maxBound.x, points[i].x);
        maxBound.y = std::max(maxBound.y, points[i].y);
        maxBound.z = std::max(maxBound.z, points[i].z);
    }

    // Create metaball field
    SDFEngine sdf;
    float threshold = 0.5f;

    sdf.setSDF([&points, radius, threshold, maxPoints](const Vec3& p) {
        float sum = 0.0f;
        for (size_t i = 0; i < maxPoints; i++) {
            float dist = (p - points[i]).length();
            if (dist < radius * 3.0f) {
                float r2 = radius * radius;
                float d2 = dist * dist;
                sum += r2 / (d2 + 0.0001f); // Metaball potential
            }
        }
        return threshold - sum;
    });

    sdf.setBounds(minBound + Vec3(-radius * 2, -radius * 2, -radius * 2),
                  maxBound + Vec3(radius * 2, radius * 2, radius * 2));

    // Use lower resolution for speed
    return MeshGenerator::marchingCubes(sdf, 20);
}
