/**
 * @file AlgorithmRegistry.h
 * @brief Qt-based algorithm metadata registry for UI population
 *
 * This header provides the AlgorithmRegistry class which maintains metadata
 * for all algorithms in Qt-compatible formats. It's used by the Qt UI layer
 * to populate the algorithm list widget with searchable, categorized algorithms.
 *
 * Key features:
 * - 127+ algorithm metadata entries with names, categories, and descriptions
 * - Automatic alphabetical sorting within categories
 * - Category descriptions for tooltips
 * - Qt QString-based for seamless Qt integration
 *
 * This registry mirrors the AlgorithmLibrary but uses Qt types for UI purposes.
 * The algorithmIndex maps to AlgorithmLibrary::AlgorithmType enum values.
 *
 * Copyright © 2025 Linus Suter
 * Released under the GNU/GPL License
 */

#pragma once

#include <QString>
#include <QStringList>
#include <QMap>
#include <algorithm>

/**
 * @brief Metadata for a single algorithm
 *
 * Contains UI-displayable information about an algorithm including
 * name, category, description, and index mapping to AlgorithmType enum.
 */
struct AlgorithmMetadata {
    QString name;          ///< Algorithm display name
    QString category;      ///< Category name (e.g., "PRIMITIVES", "FRACTALS")
    QString description;   ///< Tooltip description text
    int algorithmIndex;    ///< Maps to AlgorithmLibrary::AlgorithmType enum
};

/**
 * @brief Registry of algorithm metadata for UI
 *
 * Provides static methods to query algorithm metadata for populating
 * Qt UI widgets. All methods are static as this is a stateless registry.
 */
class AlgorithmRegistry {
public:
    /**
     * @brief Gets metadata for all algorithms
     * @return List of algorithm metadata sorted by category, then alphabetically
     *
     * Returns complete metadata for all 127+ algorithms. Algorithms are
     * automatically sorted alphabetically within each category.
     */
    static QList<AlgorithmMetadata> getAllAlgorithms() {
        QList<AlgorithmMetadata> algorithms;

        // PRIMITIVES (0-24) - Most common primitives listed first for easy access
        algorithms.append({"Plane", "PRIMITIVES", "Flat plane surface with subdivisions", 5});
        algorithms.append({"Cube", "PRIMITIVES", "Standard cubic primitive", 1});
        algorithms.append({"Sphere", "PRIMITIVES", "UV-mapped spherical mesh with uniform tessellation", 0});
        algorithms.append({"Cylinder", "PRIMITIVES", "Cylindrical primitive", 3});
        algorithms.append({"Cone", "PRIMITIVES", "Tapered cone shape", 4});
        algorithms.append({"Torus", "PRIMITIVES", "Donut-shaped surface", 2});
        algorithms.append({"Pyramid", "PRIMITIVES", "Square-based pyramid with apex point", 12});
        algorithms.append({"Prism", "PRIMITIVES", "Polygonal prism with customizable base", 22});
        algorithms.append({"Superellipsoid", "PRIMITIVES", "Generalized ellipsoid with adjustable squareness", 6});
        algorithms.append({"Supertoroid", "PRIMITIVES", "Torus with adjustable squareness parameters", 7});
        algorithms.append({"Teapot", "PRIMITIVES", "Authentic Utah teapot - iconic computer graphics test model", 8});
        algorithms.append({"Oil Tank", "PRIMITIVES", "Industrial oil storage tank with cylindrical body", 9});
        algorithms.append({"GeoSphere", "PRIMITIVES", "Geodesic sphere with uniform triangle distribution", 10});
        algorithms.append({"Tube", "PRIMITIVES", "Hollow cylindrical tube with adjustable wall thickness", 11});
        algorithms.append({"Hedra", "PRIMITIVES", "General polyhedron generator for Platonic solids", 13});
        algorithms.append({"Torus Knot", "PRIMITIVES", "Parametric (p,q)-torus knot with mathematical topology", 14});
        algorithms.append({"Capsule", "PRIMITIVES", "Pill-shaped capsule with hemispherical caps", 15});
        algorithms.append({"Spindle", "PRIMITIVES", "Double cone shape joined at bases", 16});
        algorithms.append({"L-Ext", "PRIMITIVES", "L-shaped extrusion profile", 17});
        algorithms.append({"C-Ext", "PRIMITIVES", "C-shaped channel extrusion", 18});
        algorithms.append({"Gengon", "PRIMITIVES", "General polygon extrusion with filleted edges", 19});
        algorithms.append({"RingWave", "PRIMITIVES", "Torus with sinusoidal wave deformation", 20});
        algorithms.append({"Hose", "PRIMITIVES", "Flexible bent tube with controllable bend angle", 21});

        // MODIFIERS (94-122) - Most common modifiers listed first for easy access
        algorithms.append({"Array", "MODIFIERS", "Array modifier creating multiple copies", 113});
        algorithms.append({"Bend Deform", "MODIFIERS", "Bend deformation with controllable angle", 107});
        algorithms.append({"Displace", "MODIFIERS", "Displace vertices using procedural texture", 115});
        algorithms.append({"Extrude", "MODIFIERS", "Extrude faces along normals", 117});
        algorithms.append({"Inflate", "MODIFIERS", "Inflate mesh along vertex normals", 102});
        algorithms.append({"Lattice Deform", "MODIFIERS", "Lattice-based FFD deformation", 111});
        algorithms.append({"Mirror", "MODIFIERS", "Mirror geometry across axis plane", 112});
        algorithms.append({"Noise", "MODIFIERS", "Apply random noise displacement to vertices", 97});
        algorithms.append({"Perlin Noise", "MODIFIERS", "Apply Perlin-like noise displacement", 98});
        algorithms.append({"Pulse", "MODIFIERS", "Radial pulse deformation from center", 99});
        algorithms.append({"Rotate", "MODIFIERS", "Rotation transformation around X, Y, Z axes", 96});
        algorithms.append({"Scale", "MODIFIERS", "Scale transformation - uniform or non-uniform sizing", 94});
        algorithms.append({"Shell", "MODIFIERS", "Shell/offset surface creating hollow forms", 114});
        algorithms.append({"Smooth", "MODIFIERS", "Laplacian smoothing to reduce roughness", 101});
        algorithms.append({"Solidify", "MODIFIERS", "Add thickness to surface mesh", 103});
        algorithms.append({"Spherify", "MODIFIERS", "Push vertices toward spherical shape", 110});
        algorithms.append({"Subdivide", "MODIFIERS", "Subdivide mesh faces for higher detail", 104});
        algorithms.append({"Taper Deform", "MODIFIERS", "Taper deformation scaling along axis", 108});
        algorithms.append({"Translate", "MODIFIERS", "Translation transformation in 3D space", 95});
        algorithms.append({"Twist Deform", "MODIFIERS", "Twist deformation along axis", 106});
        algorithms.append({"Wave Deform", "MODIFIERS", "Sinusoidal wave deformation", 109});
        algorithms.append({"Repeat Domain", "MODIFIERS", "Repeat geometry in 3D space with domain repetition", 100});
        algorithms.append({"Decimate", "MODIFIERS", "Reduce mesh complexity and polygon count", 105});
        algorithms.append({"Remesh", "MODIFIERS", "Remesh with uniform topology", 116});
        algorithms.append({"Voxelize", "MODIFIERS", "Convert mesh to voxel representation", 118});
        algorithms.append({"Laplacian Relax", "MODIFIERS", "Mesh smoothing using Laplacian relaxation", 119});
        algorithms.append({"CMA-ES Optimize", "MODIFIERS", "Optimize mesh shape using CMA-ES evolutionary strategy", 120});
        algorithms.append({"NSGA-II Multi-Objective", "MODIFIERS", "Multi-objective optimization using NSGA-II", 121});
        algorithms.append({"RBF Surrogate", "MODIFIERS", "Surrogate-based optimization using radial basis functions", 122});

        // PLATONIC SOLIDS (23-27)
        algorithms.append({"Tetrahedron", "PLATONIC SOLIDS", "Regular tetrahedron - 4 triangular faces", 23});
        algorithms.append({"Octahedron", "PLATONIC SOLIDS", "Regular octahedron - 8 triangular faces", 24});
        algorithms.append({"Icosahedron", "PLATONIC SOLIDS", "Regular icosahedron - 20 triangular faces", 25});
        algorithms.append({"Dodecahedron", "PLATONIC SOLIDS", "Regular dodecahedron - 12 pentagonal faces", 26});
        algorithms.append({"Geodesic Sphere", "PLATONIC SOLIDS", "Icosahedron-based sphere subdivision", 27});

        // PARAMETRIC SURFACES (28-36)
        algorithms.append({"Spiral Sphere", "PARAMETRIC SURFACES", "Sphere with spiral pattern creating undulating surfaces", 28});
        algorithms.append({"Twisted Torus", "PARAMETRIC SURFACES", "Torus with twist deformation", 29});
        algorithms.append({"Wave Surface", "PARAMETRIC SURFACES", "Sinusoidal wave surface with controllable frequency", 30});
        algorithms.append({"Spring", "PARAMETRIC SURFACES", "Helical coil following spring geometry", 31});
        algorithms.append({"Lissajous", "PARAMETRIC SURFACES", "3D Lissajous creating harmonic patterns", 32});
        algorithms.append({"Rose", "PARAMETRIC SURFACES", "3D rose curve with petal structure", 33});
        algorithms.append({"Helix", "PARAMETRIC SURFACES", "3D helical curve spiraling around axis", 34});
        algorithms.append({"Lathe", "PARAMETRIC SURFACES", "Revolve profile curve around axis to create surface of revolution", 35});
        algorithms.append({"Loft", "PARAMETRIC SURFACES", "Create surface by blending between multiple profile curves", 36});

        // TOPOLOGY (37-41)
        algorithms.append({"Möbius Strip", "TOPOLOGY", "Non-orientable surface with single side", 37});
        algorithms.append({"Klein Bottle", "TOPOLOGY", "4D surface in 3D space with no inside/outside", 38});
        algorithms.append({"Trefoil Knot", "TOPOLOGY", "Simplest non-trivial knot topology", 39});
        algorithms.append({"Figure-Eight Knot", "TOPOLOGY", "Classic knot with four crossings", 40});
        algorithms.append({"Heart", "TOPOLOGY", "Mathematical heart curve in 3D", 41});

        // MINIMAL SURFACES (42-48, 123-125)
        algorithms.append({"Hyperbolic Paraboloid", "MINIMAL SURFACES", "Saddle surface with negative curvature", 42});
        algorithms.append({"Gyroid (Surfels)", "MINIMAL SURFACES", "Triply periodic minimal surface for 3D printing", 43});
        algorithms.append({"Gyroid (Marching Cubes)", "MINIMAL SURFACES", "Gyroid surface using marching cubes algorithm", 123});
        algorithms.append({"Schwarz P (Surfels)", "MINIMAL SURFACES", "Periodic minimal surface with cubic symmetry", 44});
        algorithms.append({"Schwarz P (Marching Cubes)", "MINIMAL SURFACES", "Schwarz P surface using marching cubes algorithm", 124});
        algorithms.append({"Neovius (Surfels)", "MINIMAL SURFACES", "Combines properties of gyroid and Schwarz surfaces", 45});
        algorithms.append({"Neovius (Marching Cubes)", "MINIMAL SURFACES", "Neovius surface using marching cubes algorithm", 125});
        algorithms.append({"Helicoid", "MINIMAL SURFACES", "Ruled minimal surface with helical motion", 46});
        algorithms.append({"Catenoid", "MINIMAL SURFACES", "Minimal surface between two circles", 47});
        algorithms.append({"Enneper", "MINIMAL SURFACES", "Minimal surface with self-intersections", 48});

        // ORGANIC (49-55)
        algorithms.append({"Nautilus Shell", "ORGANIC", "Logarithmic spiral shell based on golden ratio", 49});
        algorithms.append({"Conch Shell", "ORGANIC", "Conical shell with increasing radius", 50});
        algorithms.append({"Flower", "ORGANIC", "Procedural flower with customizable petals", 51});
        algorithms.append({"Coral", "ORGANIC", "Organic coral-like structure with random growth", 52});
        algorithms.append({"Crystal", "ORGANIC", "Geometric crystal structure with facets", 53});
        algorithms.append({"Tentacle", "ORGANIC", "Organic tentacle with segmented structure", 54});
        algorithms.append({"Leaf Venation", "ORGANIC", "Leaf structure with vein pattern network", 55});

        // FRACTALS (56-67)
        algorithms.append({"Fractal Tree", "FRACTALS", "Recursive branching structure mimicking trees", 56});
        algorithms.append({"L-System Plant", "FRACTALS", "Fractal plant using Lindenmayer systems", 57});
        algorithms.append({"Fractal Mountain", "FRACTALS", "Diamond-square algorithm for terrain", 58});
        algorithms.append({"Barnsley Fern", "FRACTALS", "Iterated function system fern", 59});
        algorithms.append({"Mandelbrot 2D", "FRACTALS", "2D Mandelbrot set as height field visualization", 60});
        algorithms.append({"Mandelbrot 3D", "FRACTALS", "3D Mandelbrot set fractal visualization", 61});
        algorithms.append({"Menger Sponge", "FRACTALS", "3D Cantor set demonstrating fractal dimension", 62});
        algorithms.append({"Julia Set 3D", "FRACTALS", "3D Julia set with quaternion math", 63});
        algorithms.append({"Julia Set 2D", "FRACTALS", "2D Julia set complex plane visualization", 64});
        algorithms.append({"Sierpinski", "FRACTALS", "Sierpinski triangle or pyramid fractal", 65});
        algorithms.append({"Koch Snowflake", "FRACTALS", "Koch snowflake curve fractal", 66});
        algorithms.append({"Dragon Curve", "FRACTALS", "Heighway dragon fractal in 3D", 67});

        // ARCHITECTURAL (68-72)
        algorithms.append({"Building", "ARCHITECTURAL", "Parametric building with floors and setbacks", 68});
        algorithms.append({"Voronoi Lattice", "ARCHITECTURAL", "3D Voronoi cell structure for organic patterns", 69});
        algorithms.append({"Hexagonal Grid", "ARCHITECTURAL", "Hexagonal tessellation with honeycomb structure", 70});
        algorithms.append({"Facade Pattern", "ARCHITECTURAL", "Parametric facade panels with depth variation", 71});
        algorithms.append({"Structural Frame", "ARCHITECTURAL", "3D structural grid framework for buildings", 72});

        // MECHANICAL (73-75)
        algorithms.append({"Gear", "MECHANICAL", "Mechanical gear with involute tooth profile", 73});
        algorithms.append({"Threaded Bolt", "MECHANICAL", "Bolt with helical threading for fasteners", 74});
        algorithms.append({"Bearing", "MECHANICAL", "Ball bearing structure for mechanical parts", 75});

        // STRUCTURAL (76-78)
        algorithms.append({"Diamond Lattice", "STRUCTURAL", "3D lattice based on diamond crystal structure", 76});
        algorithms.append({"Pipe Network", "STRUCTURAL", "Branching pipe system with interconnected tubes", 77});
        algorithms.append({"Lattice", "STRUCTURAL", "Regular 3D lattice framework for engineering", 78});

        // TESSELLATIONS (79-82)
        algorithms.append({"Penrose Tiling", "TESSELLATIONS", "Aperiodic tiling pattern with fivefold symmetry", 79});
        algorithms.append({"Islamic Pattern", "TESSELLATIONS", "Geometric Islamic art with radial symmetry", 80});
        algorithms.append({"Truchet Tiles", "TESSELLATIONS", "Procedural tile patterns with random rotation", 81});
        algorithms.append({"Modular Circles", "TESSELLATIONS", "Mathematical circle patterns with scaling", 82});

        // COMPUTATIONAL (83-92)
        algorithms.append({"Reaction-Diffusion", "COMPUTATIONAL", "Pattern formation using Turing equations", 83});
        algorithms.append({"Cellular Automata", "COMPUTATIONAL", "3D cellular automaton evolution system", 84});
        algorithms.append({"Voronoi Surface", "COMPUTATIONAL", "Voronoi tessellation creating cellular patterns", 85});
        algorithms.append({"Delaunay Surface", "COMPUTATIONAL", "Delaunay triangulation for smooth surfaces", 86});
        algorithms.append({"Voronoi Diagram", "COMPUTATIONAL", "2D Voronoi diagram with cell visualization", 87});
        algorithms.append({"Delaunay Triangulation", "COMPUTATIONAL", "2D Delaunay triangulation mesh", 88});
        algorithms.append({"Perlin Noise Surface", "COMPUTATIONAL", "Terrain-like surface using Perlin noise", 89});
        algorithms.append({"Neural Network", "COMPUTATIONAL", "Neural network-generated parametric surface", 90});
        algorithms.append({"CPPN-HyperNEAT", "COMPUTATIONAL", "Compositional pattern-producing network mesh generation", 91});
        algorithms.append({"Evolutionary Shape", "COMPUTATIONAL", "Evolutionary algorithm for shape optimization", 92});

        // ATTRACTORS (93)
        algorithms.append({"Lorenz Attractor", "ATTRACTORS", "Chaotic system visualization of butterfly effect", 93});

        // POINT CLOUD (126-140)
        algorithms.append({"Random Scatter", "POINT CLOUD", "Random point distribution in 3D space", 126});
        algorithms.append({"Grid Sample", "POINT CLOUD", "Uniform grid point sampling", 127});
        algorithms.append({"Poisson Disk", "POINT CLOUD", "Poisson disk sampling for even distribution", 128});
        algorithms.append({"Blue Noise", "POINT CLOUD", "Blue noise sampling for high-quality distribution", 129});
        algorithms.append({"Surface Sample", "POINT CLOUD", "Sample points on surface of geometry", 130});
        algorithms.append({"Volume Sample", "POINT CLOUD", "Sample points within volume", 131});
        algorithms.append({"Fibonacci Sphere", "POINT CLOUD", "Fibonacci spiral sphere point distribution", 132});
        algorithms.append({"Halton Sequence", "POINT CLOUD", "Low-discrepancy Halton sequence sampling", 133});
        algorithms.append({"Sobol Sequence", "POINT CLOUD", "Quasi-random Sobol sequence sampling", 134});
        algorithms.append({"Stratified Sample", "POINT CLOUD", "Stratified random sampling for uniform coverage", 135});
        algorithms.append({"Sphere Packing", "POINT CLOUD", "Dense sphere packing point generation", 136});
        algorithms.append({"Voxel Downsample", "POINT CLOUD", "Reduce point density using voxel grid", 137});
        algorithms.append({"Normal Estimate", "POINT CLOUD", "Estimate normals from point cloud", 138});
        algorithms.append({"Outlier Removal", "POINT CLOUD", "Remove statistical outliers from point cloud", 139});
        algorithms.append({"ICP Align", "POINT CLOUD", "Iterative Closest Point alignment algorithm", 140});

        // Sort algorithms alphabetically within each category (except PRIMITIVES and MODIFIERS which have custom order)
        // Categories maintain their original order
        QStringList categoryOrder = getCategories();

        // Group algorithms by category
        QMap<QString, QList<AlgorithmMetadata>> categorizedAlgorithms;
        for (const auto& algo : algorithms) {
            categorizedAlgorithms[algo.category].append(algo);
        }

        // Sort each category alphabetically (except PRIMITIVES and MODIFIERS) and rebuild the list
        algorithms.clear();
        for (const QString& category : categoryOrder) {
            if (categorizedAlgorithms.contains(category)) {
                QList<AlgorithmMetadata> categoryAlgos = categorizedAlgorithms[category];

                // PRIMITIVES and MODIFIERS categories maintain custom order (most common items first)
                // All other categories are sorted alphabetically
                if (category != "PRIMITIVES" && category != "MODIFIERS") {
                    std::sort(categoryAlgos.begin(), categoryAlgos.end(),
                        [](const AlgorithmMetadata& a, const AlgorithmMetadata& b) {
                            return a.name < b.name;
                        });
                }

                algorithms.append(categoryAlgos);
            }
        }

        return algorithms;
    }

    /**
     * @brief Gets ordered list of all categories
     * @return QStringList containing category names in display order
     *
     * Returns categories in the order they should be displayed in the UI.
     * Used by the sorting algorithm in getAllAlgorithms().
     */
    static QStringList getCategories() {
        return QStringList()
            << "PRIMITIVES"
            << "MODIFIERS"
            << "PLATONIC SOLIDS"
            << "PARAMETRIC SURFACES"
            << "TOPOLOGY"
            << "MINIMAL SURFACES"
            << "ORGANIC"
            << "FRACTALS"
            << "ARCHITECTURAL"
            << "MECHANICAL"
            << "STRUCTURAL"
            << "TESSELLATIONS"
            << "COMPUTATIONAL"
            << "ATTRACTORS"
            << "POINT CLOUD";
    }

    /**
     * @brief Gets descriptive text for a category
     * @param category Category name to query
     * @return Description string for the category, or empty string if not found
     *
     * Returns tooltip-friendly descriptions for each algorithm category.
     */
    static QString getCategoryDescription(const QString& category) {
        QMap<QString, QString> descriptions;
        descriptions["PRIMITIVES"] = "Basic geometric primitives and fundamental shapes";
        descriptions["PLATONIC SOLIDS"] = "Regular polyhedra and geodesic structures";
        descriptions["PARAMETRIC SURFACES"] = "Mathematical parametric surfaces and equations";
        descriptions["TOPOLOGY"] = "Non-orientable surfaces and knot theory";
        descriptions["MINIMAL SURFACES"] = "Zero mean curvature surfaces for advanced manufacturing";
        descriptions["ORGANIC"] = "Nature-inspired and biomimetic forms";
        descriptions["FRACTALS"] = "Self-similar recursive fractal structures";
        descriptions["ARCHITECTURAL"] = "Building design and architectural elements";
        descriptions["MECHANICAL"] = "Engineering components and mechanical parts";
        descriptions["STRUCTURAL"] = "Engineering structures and lattice frameworks";
        descriptions["TESSELLATIONS"] = "Periodic and aperiodic tiling patterns";
        descriptions["COMPUTATIONAL"] = "Algorithmic generation and simulation";
        descriptions["ATTRACTORS"] = "Chaotic systems and strange attractors";
        descriptions["MODIFIERS"] = "Mesh modification and transformation operations";
        descriptions["POINT CLOUD"] = "Point cloud generation and processing algorithms";

        return descriptions.value(category, "");
    }
};
