/**
 * @file AlgorithmLibrary.h
 * @brief Central registry for 127+ parametric mesh generation algorithms
 *
 * This header defines the AlgorithmLibrary class which manages the entire
 * collection of parametric algorithms available in the application. Each
 * algorithm is registered with parameters, descriptions, and a generator function.
 *
 * Key features:
 * - 127+ parametric mesh generation algorithms
 * - Organized into 15 categories (Primitives, Fractals, Minimal Surfaces, etc.)
 * - Dynamic parameter system with min/max/step validation
 * - Function pointer-based algorithm dispatch
 * - Supports both mesh generators and modifiers
 *
 * Algorithm Categories:
 * - Primitives (25): Basic shapes like sphere, cube, torus
 * - Platonic Solids (5): Regular polyhedra
 * - Parametric Surfaces (9): Mathematical surfaces
 * - Topology (5): Non-orientable surfaces and knots
 * - Minimal Surfaces (7): Zero mean curvature surfaces
 * - Organic (7): Nature-inspired forms
 * - Fractals (12): Self-similar recursive structures
 * - Architectural (5): Building elements
 * - Mechanical (3): Engineering components
 * - Structural (3): Lattice frameworks
 * - Tessellations (4): Tiling patterns
 * - Computational (10): Algorithmic generation
 * - Attractors (1): Chaotic systems
 * - Modifiers (29): Mesh transformations
 * - Point Cloud (15): Point generation and processing
 *
 * Copyright © 2025 Linus Suter
 * Released under the GNU/GPL License
 */

#pragma once

// Application Includes
#include "../mesh/Mesh.h"
#include "../mesh/MeshGenerator.h"
#include "../sdf/SDFEngine.h"
#include "../sdf/SDFPrimitives.h"
#include "../math/Vec3.h"

// Standard Library
#include <vector>
#include <map>
#include <string>
#include <functional>

/**
 * @brief Algorithm parameter structure
 *
 * Defines a single parameter for an algorithm with value constraints.
 * Used to generate UI sliders and validate parameter values.
 */
struct AlgorithmParameter {
    std::string name;     ///< Parameter display name
    float value;          ///< Current parameter value
    float minValue;       ///< Minimum allowed value
    float maxValue;       ///< Maximum allowed value
    float step;           ///< Step size for slider increments

    /**
     * @brief Constructor - Creates a parameter with constraints
     * @param n Parameter name
     * @param v Initial value
     * @param min Minimum value
     * @param max Maximum value
     * @param s Step size (default 0.1)
     */
    AlgorithmParameter(const std::string& n, float v, float min, float max, float s = 0.1f)
        : name(n), value(v), minValue(min), maxValue(max), step(s) {}
};

/**
 * @brief Main algorithm library class
 *
 * Central registry and dispatcher for all parametric mesh generation algorithms.
 * Manages algorithm registration, parameter validation, and mesh generation.
 */
class AlgorithmLibrary {
public:
    /**
     * @brief Type alias for algorithm generator functions
     *
     * All algorithms follow this signature: take parameters, return a mesh.
     */
    using AlgorithmFunction = std::function<Mesh(const std::vector<AlgorithmParameter>&)>;

    /**
     * @brief Enumeration of all available algorithms
     *
     * Each algorithm has a unique enum value used for identification and dispatch.
     * Algorithms are grouped by category for organization (comments show ranges).
     */
    enum AlgorithmType {
        // ===== Primitives (0-24) =====
        PRIM_SPHERE,         ///< UV-mapped sphere
        PRIM_CUBE,           ///< Standard cubic primitive
        PRIM_TORUS,          ///< Donut-shaped surface
        PRIM_CYLINDER,       ///< Cylindrical primitive
        PRIM_CONE,           ///< Tapered cone
        PRIM_PLANE,          ///< Flat subdivided plane
        PRIM_SUPERELLIPSOID, ///< Generalized ellipsoid with squareness parameter
        PRIM_SUPERTOROID,    ///< Torus with squareness parameters
        PRIM_UTAH_TEAPOT,    ///< Iconic Utah teapot test model
        PRIM_OIL_TANK,       ///< Industrial oil tank shape
        PRIM_GEOSPHERE,      ///< Geodesic sphere primitive
        PRIM_TUBE,           ///< Tube/pipe primitive
        PRIM_PYRAMID,        ///< Pyramid primitive
        PRIM_HEDRA,          ///< Polyhedron primitive
        PRIM_TORUS_KNOT,     ///< Torus knot primitive
        PRIM_CAPSULE,        ///< Capsule/pill primitive
        PRIM_SPINDLE,        ///< Spindle primitive
        PRIM_L_EXT,          ///< L-shaped extrusion
        PRIM_C_EXT,          ///< C-shaped extrusion
        PRIM_GENGON,         ///< Gengon (general polygon) primitive
        PRIM_RINGWAVE,       ///< Ring wave primitive
        PRIM_HOSE,           ///< Hose primitive
        PRIM_PRISM,          ///< Prism primitive

        // Platonic Solids (25-29)
        PLATO_TETRAHEDRON,
        PLATO_OCTAHEDRON,
        PLATO_ICOSAHEDRON,
        PLATO_DODECAHEDRON,
        PLATO_GEODESIC_SPHERE,

        // Parametric Surfaces (30-38)
        PARAM_SPIRAL_SPHERE,
        PARAM_TWISTED_TORUS,
        PARAM_WAVE,
        PARAM_SPRING,
        PARAM_LISSAJOUS,
        PARAM_ROSE,
        PARAM_HELIX,
        PARAM_LATHE,
        PARAM_LOFT,

        // Topology (39-43)
        TOPO_MOBIUS_STRIP,
        TOPO_KLEIN_BOTTLE,
        TOPO_TREFOIL_KNOT,
        TOPO_FIGURE_EIGHT_KNOT,
        TOPO_HEART,

        // Minimal Surfaces (44-50)
        MIN_HYPERBOLIC_PARABOLOID,
        MIN_GYROID,
        MIN_SCHWARZ_P,
        MIN_NEOVIUS,
        MIN_HELICOID,
        MIN_CATENOID,
        MIN_ENNEPER,

        // Organic (51-57)
        ORG_SHELL_NAUTILUS,
        ORG_SHELL_CONCH,
        ORG_FLOWER,
        ORG_CORAL,
        ORG_CRYSTAL,
        ORG_TENTACLE,
        ORG_LEAF_VENATION,

        // Fractals (58-69)
        FRAC_TREE,
        FRAC_L_SYSTEM,
        FRAC_MOUNTAIN,
        FRAC_FERN,
        FRAC_MANDELBROT_2D,
        FRAC_MANDELBROT_3D,
        FRAC_MENGER_SPONGE,
        FRAC_JULIA_3D,
        FRAC_JULIA_2D,
        FRAC_SIERPINSKI,
        FRAC_KOCH_SNOWFLAKE,
        FRAC_DRAGON_CURVE,

        // Architectural (70-74)
        ARCH_BUILDING,
        ARCH_VORONOI_LATTICE,
        ARCH_HEXAGONAL_GRID,
        ARCH_FACADE_PATTERN,
        ARCH_STRUCTURAL_FRAME,

        // Mechanical (75-77)
        MECH_GEAR,
        MECH_THREADED_BOLT,
        MECH_BEARING,

        // Structural (78-80)
        STRUCT_DIAMOND_LATTICE,
        STRUCT_PIPE_NETWORK,
        STRUCT_LATTICE,

        // Tessellations (81-84)
        TESS_PENROSE,
        TESS_ISLAMIC,
        TESS_TRUCHET,
        TESS_MODULAR_CIRCLES,

        // Computational (85-94)
        COMP_REACTION_DIFFUSION,
        COMP_CELLULAR_AUTOMATA,
        COMP_VORONOI_SURFACE,
        COMP_DELAUNAY_SURFACE,
        COMP_VORONOI_DIAGRAM,
        COMP_DELAUNAY_TRIANGULATION,
        COMP_PERLIN_NOISE_SURFACE,
        COMP_NEURAL_NETWORK,
        COMP_CPPN_HYPERNEAT,
        COMP_EVOLUTIONARY_SHAPE,

        // Attractors (95)
        ATTR_LORENZ,

        // Modifiers (96-124)
        MOD_SCALE,
        MOD_TRANSLATE,
        MOD_ROTATE,
        MOD_NOISE,
        MOD_PERLIN_NOISE,
        MOD_PULSE,
        MOD_REPEAT_DOMAIN,
        MOD_SMOOTH,
        MOD_INFLATE,
        MOD_SOLIDIFY,
        MOD_SUBDIVIDE,
        MOD_DECIMATE,
        MOD_TWIST_DEFORM,
        MOD_BEND_DEFORM,
        MOD_TAPER_DEFORM,
        MOD_WAVE_DEFORM,
        MOD_SPHERIFY,
        MOD_LATTICE_DEFORM,
        MOD_MIRROR,
        MOD_ARRAY,
        MOD_SHELL,
        MOD_DISPLACE,
        MOD_REMESH,
        MOD_EXTRUDE,
        MOD_VOXELIZE,
        MOD_LAPLACIAN_RELAX,
        MOD_CMAES_OPTIMIZE,
        MOD_NSGAII_OPTIMIZE,
        MOD_RBF_OPTIMIZE,

        // Marching Cubes variants (125-127)
        MIN_GYROID_MC,
        MIN_SCHWARZ_P_MC,
        MIN_NEOVIUS_MC,

        // Point Cloud (128-142)
        PC_RANDOM_SCATTER,
        PC_GRID_SAMPLE,
        PC_POISSON_DISK,
        PC_BLUE_NOISE,
        PC_SURFACE_SAMPLE,
        PC_VOLUME_SAMPLE,
        PC_FIBONACCI_SPHERE,
        PC_HALTON_SEQUENCE,
        PC_SOBOL_SEQUENCE,
        PC_STRATIFIED_SAMPLE,
        PC_SPHERE_PACKING,
        PC_VOXEL_DOWNSAMPLE,
        PC_NORMAL_ESTIMATE,
        PC_OUTLIER_REMOVAL,
        PC_ICP_ALIGN
    };

    /**
     * @brief Algorithm registration structure
     *
     * Contains all metadata and the generator function for a single algorithm.
     * Registered algorithms are stored in a map for fast lookup by type.
     */
    struct Algorithm {
        std::string name;                           ///< Display name (shown in UI)
        std::string description;                    ///< Tooltip description
        AlgorithmType type;                         ///< Unique algorithm identifier
        std::vector<AlgorithmParameter> parameters; ///< Algorithm parameters with defaults
        AlgorithmFunction function;                 ///< Generator function pointer
    };

private:
    std::map<AlgorithmType, Algorithm> algorithms;  ///< Registry of all algorithms

public:
    /**
     * @brief Constructor - Registers all 127+ algorithms
     *
     * Initializes the algorithm registry by registering each algorithm
     * with its name, description, parameters, and generator function.
     */
    AlgorithmLibrary();

    /**
     * @brief Generates a mesh using the specified algorithm
     * @param type Algorithm to execute
     * @param params Parameter values to use
     * @return Generated mesh
     *
     * Main entry point for mesh generation. Looks up the algorithm
     * by type and calls its generator function with the provided parameters.
     */
    Mesh generate(AlgorithmType type, const std::vector<AlgorithmParameter>& params);

    /**
     * @brief Gets algorithm metadata by type
     * @param type Algorithm to query
     * @return Algorithm structure containing name, description, parameters, and function
     *
     * Returns a reference to the Algorithm struct for the given type.
     * Used by UI to populate parameter controls and descriptions.
     */
    const Algorithm& getAlgorithm(AlgorithmType type) const {
        auto it = algorithms.find(type);
        if (it != algorithms.end()) {
            return it->second;
        }
        // Return a default algorithm if not found (should never happen)
        static Algorithm defaultAlgo = {"Unknown", "Unknown algorithm", PRIM_SPHERE, {}, nullptr};
        return defaultAlgo;
    }

    /**
     * @brief Gets list of all algorithm names
     * @return Vector of algorithm names
     */
    std::vector<std::string> getAllAlgorithmNames() const;

    /**
     * @brief Gets list of all registered algorithm types
     * @return Vector of AlgorithmType enums
     */
    std::vector<AlgorithmType> getAllRegisteredTypes() const;

    // ===== Static Generator Functions =====
    // Each algorithm has a static generator function that takes parameters
    // and returns a generated mesh. Functions are grouped by category below.
    static Mesh generateSpiralSphere(const std::vector<AlgorithmParameter>& params);
    static Mesh generateTwistedTorus(const std::vector<AlgorithmParameter>& params);
    static Mesh generateGeodesicDome(const std::vector<AlgorithmParameter>& params);
    static Mesh generateParametricWave(const std::vector<AlgorithmParameter>& params);
    static Mesh generateLathe(const std::vector<AlgorithmParameter>& params);
    static Mesh generateLoft(const std::vector<AlgorithmParameter>& params);
    static Mesh generateShellStructure(const std::vector<AlgorithmParameter>& params);

    static Mesh generateVoronoiLattice(const std::vector<AlgorithmParameter>& params);
    static Mesh generateHexagonalGrid(const std::vector<AlgorithmParameter>& params);
    static Mesh generateBuildingMassing(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFacadePattern(const std::vector<AlgorithmParameter>& params);
    static Mesh generateStructuralFrame(const std::vector<AlgorithmParameter>& params);

    static Mesh generateGearSystem(const std::vector<AlgorithmParameter>& params);
    static Mesh generateSpringCoil(const std::vector<AlgorithmParameter>& params);
    static Mesh generateThreadedBolt(const std::vector<AlgorithmParameter>& params);
    static Mesh generateBearingAssembly(const std::vector<AlgorithmParameter>& params);

    static Mesh generateBranchingTree(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCoralGrowth(const std::vector<AlgorithmParameter>& params);
    static Mesh generateShellGrowth(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFlowerPetals(const std::vector<AlgorithmParameter>& params);
    static Mesh generateLeafVenation(const std::vector<AlgorithmParameter>& params);

    static Mesh generateMobiusStrip(const std::vector<AlgorithmParameter>& params);
    static Mesh generateKleinBottle(const std::vector<AlgorithmParameter>& params);
    static Mesh generateTrefoilKnot(const std::vector<AlgorithmParameter>& params);
    static Mesh generateLorenzAttractor(const std::vector<AlgorithmParameter>& params);
    static Mesh generateHyperbolicParaboloid(const std::vector<AlgorithmParameter>& params);

    static Mesh generatePenroseTiling(const std::vector<AlgorithmParameter>& params);
    static Mesh generateIslamicPattern(const std::vector<AlgorithmParameter>& params);
    static Mesh generateTruchetTiles(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModularCircles(const std::vector<AlgorithmParameter>& params);

    static Mesh generateGyroidSurface(const std::vector<AlgorithmParameter>& params);
    static Mesh generateSchwarzPSurface(const std::vector<AlgorithmParameter>& params);
    static Mesh generateReactionDiffusion(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCellularAutomata(const std::vector<AlgorithmParameter>& params);
    static Mesh generateLSystemPlant(const std::vector<AlgorithmParameter>& params);

    // Modifiers
    static Mesh generateModScale(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModTranslate(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModRotate(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModNoise(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModPerlinNoise(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModPulse(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModRepeatDomain(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModSmooth(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModInflate(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModSolidify(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModSubdivide(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModDecimate(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModTwistDeform(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModBendDeform(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModTaperDeform(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModWaveDeform(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModSpherify(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModLatticeDeform(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModMirror(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModArray(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModShell(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModDisplace(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModRemesh(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModExtrude(const std::vector<AlgorithmParameter>& params);

    // Additional algorithm declarations (stubs - use existing MeshGenerator functions)
    static Mesh generatePrimSphere(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimCube(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimTorus(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimCylinder(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimCone(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimPlane(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimSuperellipsoid(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimSupertoroid(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimUtahTeapot(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimOilTank(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimGeosphere(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimTube(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimPyramid(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimHedra(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimTorusKnot(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimCapsule(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimSpindle(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimLExt(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimCExt(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimGengon(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimRingWave(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimHose(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePrimPrism(const std::vector<AlgorithmParameter>& params);

    // Platonic solids
    static Mesh generatePlatoTetrahedron(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePlatoOctahedron(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePlatoIcosahedron(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePlatoDodecahedron(const std::vector<AlgorithmParameter>& params);

    // Parametric curves
    static Mesh generateParamLissajous(const std::vector<AlgorithmParameter>& params);
    static Mesh generateParamRose(const std::vector<AlgorithmParameter>& params);
    static Mesh generateParamHelix(const std::vector<AlgorithmParameter>& params);

    // Topology
    static Mesh generateTopoFigureEightKnot(const std::vector<AlgorithmParameter>& params);
    static Mesh generateTopoHeart(const std::vector<AlgorithmParameter>& params);

    // Minimal surfaces
    static Mesh generateMinNeovius(const std::vector<AlgorithmParameter>& params);
    static Mesh generateMinHelicoid(const std::vector<AlgorithmParameter>& params);
    static Mesh generateMinCatenoid(const std::vector<AlgorithmParameter>& params);
    static Mesh generateMinEnneper(const std::vector<AlgorithmParameter>& params);

    // Organic
    static Mesh generateOrgShellConch(const std::vector<AlgorithmParameter>& params);
    static Mesh generateOrgCrystal(const std::vector<AlgorithmParameter>& params);
    static Mesh generateOrgTentacle(const std::vector<AlgorithmParameter>& params);

    // Fractals
    static Mesh generateFracMountain(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracFern(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracMandelbrot2D(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracMandelbrot3D(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracMengerSponge(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracJulia3D(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracJulia2D(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracSierpinski(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracKochSnowflake(const std::vector<AlgorithmParameter>& params);
    static Mesh generateFracDragonCurve(const std::vector<AlgorithmParameter>& params);

    // Structural
    static Mesh generateStructDiamondLattice(const std::vector<AlgorithmParameter>& params);
    static Mesh generateStructPipeNetwork(const std::vector<AlgorithmParameter>& params);
    static Mesh generateStructLattice(const std::vector<AlgorithmParameter>& params);

    // Computational
    static Mesh generateCompVoronoiSurface(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCompDelaunaySurface(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCompVoronoiDiagram(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCompDelaunayTriangulation(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCompPerlinNoiseSurface(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCompNeuralNetwork(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCompCPPNHyperNEAT(const std::vector<AlgorithmParameter>& params);
    static Mesh generateCompEvolutionaryShape(const std::vector<AlgorithmParameter>& params);

    // Additional modifiers
    static Mesh generateModVoxelize(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModLaplacianRelax(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModCMAESOptimize(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModNSGAIIOptimize(const std::vector<AlgorithmParameter>& params);
    static Mesh generateModRBFOptimize(const std::vector<AlgorithmParameter>& params);

    // Marching Cubes variants
    static Mesh generateGyroidSurfaceMC(const std::vector<AlgorithmParameter>& params);
    static Mesh generateSchwarzPSurfaceMC(const std::vector<AlgorithmParameter>& params);
    static Mesh generateMinNeoviusMC(const std::vector<AlgorithmParameter>& params);

    // Point Cloud
    static Mesh generatePCRandomScatter(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCGridSample(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCPoissonDisk(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCBlueNoise(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCSurfaceSample(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCVolumeSample(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCFibonacciSphere(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCHaltonSequence(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCSobolSequence(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCStratifiedSample(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCSpherePacking(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCVoxelDownsample(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCNormalEstimate(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCOutlierRemoval(const std::vector<AlgorithmParameter>& params);
    static Mesh generatePCICPAlign(const std::vector<AlgorithmParameter>& params);
};
