#include "AlgorithmLibrary.h"
#include "FractalGenerators.h"
#include "PatternGenerator.h"
#include "PointCloudMeshing.h"
#include "../voxel/VoxelGenerator.h"
#include "../neuralnet/NeuralNetwork.h"
#include "../neuralnet/CPPNHyperNEAT.h"
#include "../optimization/CMAESOptimizer.h"
#include "../optimization/NSGAIIOptimizer.h"
#include "../optimization/RBFOptimizer.h"
#include <QApplication>
#include <cmath>
#include <random>
#include <functional>

AlgorithmLibrary::AlgorithmLibrary() {
    // Parametric shapes
    algorithms[PARAM_SPIRAL_SPHERE] = {
        "Spiral Sphere",
        "Sphere with spiral pattern",
        PARAM_SPIRAL_SPHERE,
        {
            AlgorithmParameter("Radius", 1.0f, 0.1f, 5.0f),
            AlgorithmParameter("Spirals", 8.0f, 1.0f, 20.0f, 1.0f),
            AlgorithmParameter("Turns", 3.0f, 1.0f, 10.0f, 1.0f),
            AlgorithmParameter("Resolution", 32.0f, 8.0f, 128.0f, 4.0f)
        },
        generateSpiralSphere
    };

    algorithms[PARAM_TWISTED_TORUS] = {
        "Twisted Torus",
        "Torus with twist deformation",
        PARAM_TWISTED_TORUS,
        {
            AlgorithmParameter("Major Radius", 2.0f, 0.5f, 5.0f),
            AlgorithmParameter("Minor Radius", 0.5f, 0.1f, 2.0f),
            AlgorithmParameter("Twist Amount", 2.0f, 0.0f, 10.0f),
            AlgorithmParameter("Segments", 40.0f, 8.0f, 128.0f, 4.0f)
        },
        generateTwistedTorus
    };

    algorithms[PLATO_GEODESIC_SPHERE] = {
        "Geodesic Dome",
        "Geodesic sphere subdivision",
        PLATO_GEODESIC_SPHERE,
        {
            AlgorithmParameter("Radius", 2.0f, 0.5f, 5.0f),
            AlgorithmParameter("Subdivisions", 2.0f, 0.0f, 5.0f, 1.0f),
            AlgorithmParameter("Frequency", 1.0f, 1.0f, 10.0f, 1.0f)
        },
        generateGeodesicDome
    };

    algorithms[PARAM_WAVE] = {
        "Parametric Wave Surface",
        "Mathematical wave surface",
        PARAM_WAVE,
        {
            AlgorithmParameter("Amplitude", 1.0f, 0.1f, 3.0f),
            AlgorithmParameter("Frequency X", 2.0f, 0.5f, 10.0f),
            AlgorithmParameter("Frequency Y", 2.0f, 0.5f, 10.0f),
            AlgorithmParameter("Resolution", 64.0f, 16.0f, 128.0f, 8.0f)
        },
        generateParametricWave
    };

    algorithms[ORG_SHELL_NAUTILUS] = {
        "Shell Structure",
        "Logarithmic spiral shell",
        ORG_SHELL_NAUTILUS,
        {
            AlgorithmParameter("Growth Rate", 0.2f, 0.05f, 0.5f),
            AlgorithmParameter("Turns", 3.0f, 1.0f, 8.0f),
            AlgorithmParameter("Radius", 1.0f, 0.5f, 3.0f),
            AlgorithmParameter("Segments", 64.0f, 16.0f, 128.0f, 8.0f)
        },
        generateShellStructure
    };

    algorithms[PARAM_LATHE] = {
        "Lathe",
        "Revolve profile around axis",
        PARAM_LATHE,
        {
            AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f),
            AlgorithmParameter("Profile Scale", 1.0f, 0.5f, 2.0f)
        },
        generateLathe
    };

    algorithms[PARAM_LOFT] = {
        "Loft",
        "Blend between profiles",
        PARAM_LOFT,
        {
            AlgorithmParameter("Segments", 16.0f, 4.0f, 32.0f),
            AlgorithmParameter("Blend Factor", 0.5f, 0.0f, 1.0f)
        },
        generateLoft
    };

    // Architectural
    algorithms[ARCH_VORONOI_LATTICE] = {
        "Voronoi Lattice",
        "3D Voronoi cell structure",
        ARCH_VORONOI_LATTICE,
        {
            AlgorithmParameter("Cell Count", 10.0f, 3.0f, 30.0f, 1.0f),
            AlgorithmParameter("Size", 5.0f, 2.0f, 10.0f),
            AlgorithmParameter("Thickness", 0.1f, 0.05f, 0.5f)
        },
        generateVoronoiLattice
    };

    algorithms[ARCH_HEXAGONAL_GRID] = {
        "Hexagonal Grid",
        "Hexagonal tessellation",
        ARCH_HEXAGONAL_GRID,
        {
            AlgorithmParameter("Rings", 5.0f, 1.0f, 15.0f, 1.0f),
            AlgorithmParameter("Radius", 1.0f, 0.3f, 3.0f),
            AlgorithmParameter("Height", 0.2f, 0.05f, 1.0f),
            AlgorithmParameter("Gap", 0.05f, 0.0f, 0.3f)
        },
        generateHexagonalGrid
    };

    algorithms[ARCH_BUILDING] = {
        "Building Massing",
        "Procedural building volumes",
        ARCH_BUILDING,
        {
            AlgorithmParameter("Width", 10.0f, 3.0f, 30.0f),
            AlgorithmParameter("Depth", 10.0f, 3.0f, 30.0f),
            AlgorithmParameter("Height", 20.0f, 5.0f, 50.0f),
            AlgorithmParameter("Floors", 5.0f, 1.0f, 20.0f, 1.0f),
            AlgorithmParameter("Setback", 0.8f, 0.5f, 1.0f)
        },
        generateBuildingMassing
    };

    algorithms[ARCH_FACADE_PATTERN] = {
        "Facade Pattern",
        "Parametric facade panels",
        ARCH_FACADE_PATTERN,
        {
            AlgorithmParameter("Width", 10.0f, 5.0f, 20.0f),
            AlgorithmParameter("Height", 15.0f, 5.0f, 30.0f),
            AlgorithmParameter("Panel Width", 1.0f, 0.5f, 2.0f),
            AlgorithmParameter("Panel Height", 1.5f, 0.5f, 3.0f),
            AlgorithmParameter("Depth Variation", 0.3f, 0.0f, 1.0f)
        },
        generateFacadePattern
    };

    algorithms[ARCH_STRUCTURAL_FRAME] = {
        "Structural Frame",
        "Parametric structural grid",
        ARCH_STRUCTURAL_FRAME,
        {
            AlgorithmParameter("Width", 10.0f, 5.0f, 20.0f),
            AlgorithmParameter("Height", 10.0f, 5.0f, 20.0f),
            AlgorithmParameter("Depth", 10.0f, 5.0f, 20.0f),
            AlgorithmParameter("Divisions", 4.0f, 2.0f, 10.0f, 1.0f),
            AlgorithmParameter("Member Size", 0.2f, 0.1f, 0.5f)
        },
        generateStructuralFrame
    };

    // Mechanical
    algorithms[MECH_GEAR] = {
        "Gear System",
        "Mechanical gear with teeth",
        MECH_GEAR,
        {
            AlgorithmParameter("Teeth Count", 20.0f, 6.0f, 60.0f, 1.0f),
            AlgorithmParameter("Outer Radius", 2.0f, 0.5f, 5.0f),
            AlgorithmParameter("Inner Radius", 1.5f, 0.3f, 4.0f),
            AlgorithmParameter("Thickness", 0.3f, 0.1f, 1.0f),
            AlgorithmParameter("Tooth Depth", 0.3f, 0.1f, 0.8f)
        },
        generateGearSystem
    };

    algorithms[PARAM_SPRING] = {
        "Spring Coil",
        "Helical spring structure",
        PARAM_SPRING,
        {
            AlgorithmParameter("Radius", 1.0f, 0.3f, 3.0f),
            AlgorithmParameter("Coil Radius", 0.1f, 0.05f, 0.3f),
            AlgorithmParameter("Turns", 8.0f, 2.0f, 20.0f),
            AlgorithmParameter("Pitch", 0.5f, 0.1f, 2.0f),
            AlgorithmParameter("Segments", 16.0f, 8.0f, 32.0f, 4.0f)
        },
        generateSpringCoil
    };

    algorithms[MECH_THREADED_BOLT] = {
        "Threaded Bolt",
        "Bolt with helical threads",
        MECH_THREADED_BOLT,
        {
            AlgorithmParameter("Radius", 0.5f, 0.2f, 2.0f),
            AlgorithmParameter("Length", 3.0f, 1.0f, 10.0f),
            AlgorithmParameter("Thread Pitch", 0.2f, 0.05f, 0.5f),
            AlgorithmParameter("Thread Depth", 0.1f, 0.02f, 0.3f),
            AlgorithmParameter("Segments", 40.0f, 16.0f, 64.0f, 4.0f)
        },
        generateThreadedBolt
    };

    algorithms[MECH_BEARING] = {
        "Bearing Assembly",
        "Ball bearing structure",
        MECH_BEARING,
        {
            AlgorithmParameter("Inner Radius", 1.0f, 0.5f, 3.0f),
            AlgorithmParameter("Outer Radius", 2.0f, 1.0f, 5.0f),
            AlgorithmParameter("Ball Count", 8.0f, 4.0f, 16.0f, 1.0f),
            AlgorithmParameter("Ball Radius", 0.3f, 0.1f, 0.5f),
            AlgorithmParameter("Thickness", 0.5f, 0.2f, 1.0f)
        },
        generateBearingAssembly
    };

    // Organic
    algorithms[FRAC_TREE] = {
        "Branching Tree",
        "Recursive tree structure",
        FRAC_TREE,
        {
            AlgorithmParameter("Trunk Radius", 0.3f, 0.1f, 1.0f),
            AlgorithmParameter("Branch Length", 2.0f, 0.5f, 5.0f),
            AlgorithmParameter("Recursion Depth", 4.0f, 1.0f, 7.0f, 1.0f),
            AlgorithmParameter("Branch Angle", 30.0f, 10.0f, 60.0f),
            AlgorithmParameter("Branch Count", 3.0f, 2.0f, 5.0f, 1.0f)
        },
        generateBranchingTree
    };

    algorithms[ORG_CORAL] = {
        "Coral Growth",
        "Organic coral-like structure",
        ORG_CORAL,
        {
            AlgorithmParameter("Base Radius", 0.5f, 0.2f, 2.0f),
            AlgorithmParameter("Growth Factor", 1.3f, 1.1f, 2.0f),
            AlgorithmParameter("Iterations", 5.0f, 2.0f, 8.0f, 1.0f),
            AlgorithmParameter("Randomness", 0.3f, 0.0f, 1.0f)
        },
        generateCoralGrowth
    };

    algorithms[ORG_FLOWER] = {
        "Flower Petals",
        "Parametric flower with petals",
        ORG_FLOWER,
        {
            AlgorithmParameter("Petal Count", 8.0f, 3.0f, 16.0f, 1.0f),
            AlgorithmParameter("Petal Length", 1.5f, 0.5f, 3.0f),
            AlgorithmParameter("Petal Width", 0.8f, 0.3f, 2.0f),
            AlgorithmParameter("Curvature", 0.5f, 0.0f, 1.0f)
        },
        generateFlowerPetals
    };

    algorithms[ORG_LEAF_VENATION] = {
        "Leaf Venation",
        "Leaf with vein pattern",
        ORG_LEAF_VENATION,
        {
            AlgorithmParameter("Leaf Width", 2.0f, 1.0f, 5.0f),
            AlgorithmParameter("Leaf Length", 3.0f, 1.5f, 6.0f),
            AlgorithmParameter("Vein Density", 0.5f, 0.2f, 1.0f),
            AlgorithmParameter("Branching", 3.0f, 1.0f, 5.0f, 1.0f)
        },
        generateLeafVenation
    };

    // Mathematical surfaces
    algorithms[TOPO_MOBIUS_STRIP] = {
        "Möbius Strip",
        "Non-orientable surface",
        TOPO_MOBIUS_STRIP,
        {
            AlgorithmParameter("Radius", 2.0f, 1.0f, 5.0f),
            AlgorithmParameter("Width", 1.0f, 0.3f, 2.0f),
            AlgorithmParameter("Segments", 64.0f, 16.0f, 128.0f, 8.0f)
        },
        generateMobiusStrip
    };

    algorithms[TOPO_KLEIN_BOTTLE] = {
        "Klein Bottle",
        "4D surface in 3D space",
        TOPO_KLEIN_BOTTLE,
        {
            AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f),
            AlgorithmParameter("Segments", 40.0f, 16.0f, 64.0f, 4.0f)
        },
        generateKleinBottle
    };

    algorithms[TOPO_TREFOIL_KNOT] = {
        "Trefoil Knot",
        "Mathematical knot structure",
        TOPO_TREFOIL_KNOT,
        {
            AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f),
            AlgorithmParameter("Tube Radius", 0.3f, 0.1f, 0.8f),
            AlgorithmParameter("Segments", 64.0f, 32.0f, 128.0f, 8.0f)
        },
        generateTrefoilKnot
    };

    algorithms[ATTR_LORENZ] = {
        "Lorenz Attractor",
        "Chaotic system visualization",
        ATTR_LORENZ,
        {
            AlgorithmParameter("Sigma", 10.0f, 5.0f, 15.0f),
            AlgorithmParameter("Rho", 28.0f, 20.0f, 35.0f),
            AlgorithmParameter("Beta", 2.667f, 1.0f, 5.0f),
            AlgorithmParameter("Iterations", 5000.0f, 1000.0f, 10000.0f, 100.0f),
            AlgorithmParameter("Scale", 0.1f, 0.05f, 0.3f)
        },
        generateLorenzAttractor
    };

    algorithms[MIN_HYPERBOLIC_PARABOLOID] = {
        "Hyperbolic Paraboloid",
        "Saddle surface",
        MIN_HYPERBOLIC_PARABOLOID,
        {
            AlgorithmParameter("Size", 3.0f, 1.0f, 8.0f),
            AlgorithmParameter("Curvature", 1.0f, 0.3f, 3.0f),
            AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f, 4.0f)
        },
        generateHyperbolicParaboloid
    };

    // Tessellations and patterns
    algorithms[TESS_PENROSE] = {
        "Penrose Tiling",
        "Aperiodic tiling pattern",
        TESS_PENROSE,
        {
            AlgorithmParameter("Iterations", 3.0f, 1.0f, 6.0f, 1.0f),
            AlgorithmParameter("Scale", 2.0f, 1.0f, 5.0f),
            AlgorithmParameter("Thickness", 0.1f, 0.05f, 0.3f)
        },
        generatePenroseTiling
    };

    algorithms[TESS_ISLAMIC] = {
        "Islamic Pattern",
        "Geometric Islamic art",
        TESS_ISLAMIC,
        {
            AlgorithmParameter("Symmetry", 8.0f, 4.0f, 12.0f, 1.0f),
            AlgorithmParameter("Radius", 5.0f, 2.0f, 10.0f),
            AlgorithmParameter("Complexity", 3.0f, 1.0f, 6.0f, 1.0f),
            AlgorithmParameter("Line Width", 0.1f, 0.05f, 0.3f)
        },
        generateIslamicPattern
    };

    algorithms[TESS_TRUCHET] = {
        "Truchet Tiles",
        "Procedural tile patterns",
        TESS_TRUCHET,
        {
            AlgorithmParameter("Grid Size", 10.0f, 4.0f, 20.0f, 1.0f),
            AlgorithmParameter("Tile Size", 1.0f, 0.5f, 2.0f),
            AlgorithmParameter("Randomness", 0.5f, 0.0f, 1.0f),
            AlgorithmParameter("Extrusion", 0.2f, 0.0f, 0.5f)
        },
        generateTruchetTiles
    };

    algorithms[TESS_MODULAR_CIRCLES] = {
        "Modular Circles",
        "Mathematical circle patterns",
        TESS_MODULAR_CIRCLES,
        {
            AlgorithmParameter("Base Count", 12.0f, 6.0f, 24.0f, 1.0f),
            AlgorithmParameter("Radius", 2.0f, 1.0f, 5.0f),
            AlgorithmParameter("Iterations", 3.0f, 1.0f, 6.0f, 1.0f),
            AlgorithmParameter("Scale Factor", 0.7f, 0.5f, 0.9f)
        },
        generateModularCircles
    };

    // Advanced implicit surfaces
    algorithms[MIN_GYROID] = {
        "Gyroid (Surfels)",
        "Triply periodic minimal surface",
        MIN_GYROID,
        {
            AlgorithmParameter("Scale", 2.0f, 1.0f, 5.0f),
            AlgorithmParameter("Thickness", 0.0f, -0.5f, 0.5f),
            AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f, 4.0f)
        },
        generateGyroidSurface
    };

    algorithms[MIN_SCHWARZ_P] = {
        "Schwarz P (Surfels)",
        "Periodic minimal surface",
        MIN_SCHWARZ_P,
        {
            AlgorithmParameter("Scale", 2.0f, 1.0f, 5.0f),
            AlgorithmParameter("Thickness", 0.0f, -0.5f, 0.5f),
            AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f, 4.0f)
        },
        generateSchwarzPSurface
    };

    algorithms[COMP_REACTION_DIFFUSION] = {
        "Reaction-Diffusion",
        "Pattern formation simulation",
        COMP_REACTION_DIFFUSION,
        {
            AlgorithmParameter("Grid Size", 64.0f, 32.0f, 128.0f, 16.0f),
            AlgorithmParameter("Feed Rate", 0.055f, 0.01f, 0.1f),
            AlgorithmParameter("Kill Rate", 0.062f, 0.04f, 0.08f),
            AlgorithmParameter("Iterations", 1000.0f, 100.0f, 5000.0f, 100.0f),
            AlgorithmParameter("Threshold", 0.5f, 0.1f, 0.9f)
        },
        generateReactionDiffusion
    };

    algorithms[COMP_CELLULAR_AUTOMATA] = {
        "Cellular Automata",
        "3D cellular automaton",
        COMP_CELLULAR_AUTOMATA,
        {
            AlgorithmParameter("Grid Size", 16.0f, 8.0f, 32.0f, 4.0f),
            AlgorithmParameter("Generations", 10.0f, 3.0f, 20.0f, 1.0f),
            AlgorithmParameter("Birth Rule", 4.0f, 1.0f, 8.0f, 1.0f),
            AlgorithmParameter("Survival Rule", 5.0f, 2.0f, 8.0f, 1.0f)
        },
        generateCellularAutomata
    };

    algorithms[FRAC_L_SYSTEM] = {
        "L-System Plant",
        "Fractal plant generation",
        FRAC_L_SYSTEM,
        {
            AlgorithmParameter("Iterations", 4.0f, 2.0f, 7.0f, 1.0f),
            AlgorithmParameter("Angle", 25.0f, 10.0f, 45.0f),
            AlgorithmParameter("Length Scale", 0.7f, 0.5f, 0.9f),
            AlgorithmParameter("Thickness", 0.1f, 0.05f, 0.3f)
        },
        generateLSystemPlant
    };

    // MODIFIERS
    algorithms[MOD_SCALE] = {
        "Scale", "Scale transformation", MOD_SCALE,
        {AlgorithmParameter("Scale X", 1.0f, 0.1f, 3.0f), AlgorithmParameter("Scale Y", 1.0f, 0.1f, 3.0f), AlgorithmParameter("Scale Z", 1.0f, 0.1f, 3.0f)},
        generateModScale
    };

    algorithms[MOD_TRANSLATE] = {
        "Translate", "Translation transformation", MOD_TRANSLATE,
        {AlgorithmParameter("X", 0.0f, -5.0f, 5.0f), AlgorithmParameter("Y", 0.0f, -5.0f, 5.0f), AlgorithmParameter("Z", 0.0f, -5.0f, 5.0f)},
        generateModTranslate
    };

    algorithms[MOD_ROTATE] = {
        "Rotate", "Rotation transformation", MOD_ROTATE,
        {AlgorithmParameter("Rot X", 0.0f, 0.0f, 360.0f), AlgorithmParameter("Rot Y", 0.0f, 0.0f, 360.0f), AlgorithmParameter("Rot Z", 0.0f, 0.0f, 360.0f)},
        generateModRotate
    };

    algorithms[MOD_NOISE] = {
        "Noise", "Random noise displacement", MOD_NOISE,
        {AlgorithmParameter("Amplitude", 0.2f, 0.0f, 2.0f), AlgorithmParameter("Seed", 0.0f, 0.0f, 100.0f)},
        generateModNoise
    };

    algorithms[MOD_PERLIN_NOISE] = {
        "Perlin Noise", "Perlin noise displacement", MOD_PERLIN_NOISE,
        {AlgorithmParameter("Amplitude", 0.5f, 0.0f, 2.0f), AlgorithmParameter("Frequency", 2.0f, 0.5f, 10.0f), AlgorithmParameter("Resolution", 32.0f, 8.0f, 64.0f)},
        generateModPerlinNoise
    };

    algorithms[MOD_PULSE] = {
        "Pulse", "Radial pulse deformation", MOD_PULSE,
        {AlgorithmParameter("Amplitude", 0.3f, 0.0f, 2.0f), AlgorithmParameter("Frequency", 2.0f, 0.5f, 10.0f)},
        generateModPulse
    };

    algorithms[MOD_REPEAT_DOMAIN] = {
        "Repeat Domain", "Repeat in 3D space", MOD_REPEAT_DOMAIN,
        {AlgorithmParameter("Repeat X", 3.0f, 1.0f, 10.0f, 1.0f), AlgorithmParameter("Repeat Y", 3.0f, 1.0f, 10.0f, 1.0f), AlgorithmParameter("Repeat Z", 3.0f, 1.0f, 10.0f, 1.0f), AlgorithmParameter("Spacing", 2.0f, 0.5f, 5.0f)},
        generateModRepeatDomain
    };

    algorithms[MOD_SMOOTH] = {
        "Smooth", "Laplacian smoothing", MOD_SMOOTH,
        {AlgorithmParameter("Iterations", 3.0f, 1.0f, 10.0f), AlgorithmParameter("Factor", 0.5f, 0.1f, 1.0f)},
        generateModSmooth
    };

    algorithms[MOD_INFLATE] = {
        "Inflate", "Inflate along normals", MOD_INFLATE,
        {AlgorithmParameter("Amount", 0.2f, -1.0f, 1.0f)},
        generateModInflate
    };

    algorithms[MOD_SOLIDIFY] = {
        "Solidify", "Add thickness", MOD_SOLIDIFY,
        {AlgorithmParameter("Thickness", 0.2f, 0.0f, 1.0f)},
        generateModSolidify
    };

    algorithms[MOD_SUBDIVIDE] = {
        "Subdivide", "Subdivide mesh", MOD_SUBDIVIDE,
        {AlgorithmParameter("Iterations", 1.0f, 1.0f, 3.0f)},
        generateModSubdivide
    };

    algorithms[MOD_DECIMATE] = {
        "Decimate", "Reduce polygon count", MOD_DECIMATE,
        {AlgorithmParameter("Ratio", 0.5f, 0.1f, 1.0f)},
        generateModDecimate
    };

    algorithms[MOD_TWIST_DEFORM] = {
        "Twist Deform", "Twist deformation", MOD_TWIST_DEFORM,
        {AlgorithmParameter("Amount", 1.0f, 0.0f, 5.0f)},
        generateModTwistDeform
    };

    algorithms[MOD_BEND_DEFORM] = {
        "Bend Deform", "Bend deformation", MOD_BEND_DEFORM,
        {AlgorithmParameter("Angle", 45.0f, 0.0f, 180.0f)},
        generateModBendDeform
    };

    algorithms[MOD_TAPER_DEFORM] = {
        "Taper Deform", "Taper deformation", MOD_TAPER_DEFORM,
        {AlgorithmParameter("Amount", 0.5f, 0.0f, 1.0f)},
        generateModTaperDeform
    };

    algorithms[MOD_WAVE_DEFORM] = {
        "Wave Deform", "Wave deformation", MOD_WAVE_DEFORM,
        {AlgorithmParameter("Amplitude", 0.5f, 0.0f, 2.0f), AlgorithmParameter("Frequency", 3.0f, 0.5f, 10.0f)},
        generateModWaveDeform
    };

    algorithms[MOD_SPHERIFY] = {
        "Spherify", "Push toward sphere", MOD_SPHERIFY,
        {AlgorithmParameter("Amount", 0.5f, 0.0f, 1.0f)},
        generateModSpherify
    };

    algorithms[MOD_LATTICE_DEFORM] = {
        "Lattice Deform", "Lattice FFD", MOD_LATTICE_DEFORM,
        {AlgorithmParameter("Strength", 0.2f, 0.0f, 1.0f), AlgorithmParameter("Resolution", 5.0f, 2.0f, 10.0f)},
        generateModLatticeDeform
    };

    algorithms[MOD_MIRROR] = {
        "Mirror", "Mirror across axis", MOD_MIRROR,
        {AlgorithmParameter("Axis (0=X,1=Y,2=Z)", 0.0f, 0.0f, 2.0f)},
        generateModMirror
    };

    algorithms[MOD_ARRAY] = {
        "Array", "Array modifier", MOD_ARRAY,
        {AlgorithmParameter("Count", 5.0f, 1.0f, 20.0f), AlgorithmParameter("Spacing", 1.0f, 0.1f, 5.0f)},
        generateModArray
    };

    algorithms[MOD_SHELL] = {
        "Shell", "Shell/offset surface", MOD_SHELL,
        {AlgorithmParameter("Thickness", 0.2f, 0.01f, 0.5f)},
        generateModShell
    };

    algorithms[MOD_DISPLACE] = {
        "Displace", "Texture displacement", MOD_DISPLACE,
        {AlgorithmParameter("Strength", 0.5f, 0.0f, 2.0f), AlgorithmParameter("Scale", 3.0f, 0.5f, 10.0f)},
        generateModDisplace
    };

    algorithms[MOD_REMESH] = {
        "Remesh", "Uniform remesh", MOD_REMESH,
        {AlgorithmParameter("Resolution", 16.0f, 4.0f, 64.0f)},
        generateModRemesh
    };

    algorithms[MOD_EXTRUDE] = {
        "Extrude", "Extrude faces", MOD_EXTRUDE,
        {AlgorithmParameter("Distance", 1.0f, 0.0f, 5.0f)},
        generateModExtrude
    };

    // PRIMITIVES
    algorithms[PRIM_SPHERE] = {
        "Sphere", "Basic sphere primitive", PRIM_SPHERE,
        {AlgorithmParameter("Radius", 1.0f, 0.1f, 5.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimSphere
    };

    algorithms[PRIM_CUBE] = {
        "Cube", "Standard cubic primitive", PRIM_CUBE,
        {AlgorithmParameter("Size", 1.0f, 0.1f, 5.0f)},
        generatePrimCube
    };

    algorithms[PRIM_TORUS] = {
        "Torus", "Donut-shaped surface", PRIM_TORUS,
        {AlgorithmParameter("Major Radius", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Minor Radius", 0.3f, 0.1f, 1.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimTorus
    };

    algorithms[PRIM_CYLINDER] = {
        "Cylinder", "Cylindrical primitive", PRIM_CYLINDER,
        {AlgorithmParameter("Radius", 0.5f, 0.1f, 2.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimCylinder
    };

    algorithms[PRIM_CONE] = {
        "Cone", "Tapered cone shape", PRIM_CONE,
        {AlgorithmParameter("Radius", 1.0f, 0.1f, 3.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimCone
    };

    algorithms[PRIM_PLANE] = {
        "Plane", "Flat plane surface", PRIM_PLANE,
        {AlgorithmParameter("Width", 4.0f, 1.0f, 10.0f), AlgorithmParameter("Height", 4.0f, 1.0f, 10.0f), AlgorithmParameter("Width Segments", 16.0f, 1.0f, 64.0f), AlgorithmParameter("Height Segments", 16.0f, 1.0f, 64.0f)},
        generatePrimPlane
    };

    algorithms[PRIM_SUPERELLIPSOID] = {
        "Superellipsoid", "Generalized ellipsoid", PRIM_SUPERELLIPSOID,
        {AlgorithmParameter("Radius", 1.0f, 0.5f, 3.0f), AlgorithmParameter("N", 2.0f, 0.5f, 5.0f), AlgorithmParameter("E", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimSuperellipsoid
    };

    algorithms[PRIM_SUPERTOROID] = {
        "Supertoroid", "Generalized torus", PRIM_SUPERTOROID,
        {AlgorithmParameter("Major Radius", 1.5f, 0.5f, 3.0f), AlgorithmParameter("Minor Radius", 0.5f, 0.1f, 1.0f), AlgorithmParameter("N", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimSupertoroid
    };

    algorithms[PRIM_UTAH_TEAPOT] = {
        "Teapot", "Authentic Utah teapot", PRIM_UTAH_TEAPOT,
        {AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Subdivisions", 8.0f, 4.0f, 32.0f)},
        generatePrimUtahTeapot
    };

    algorithms[PRIM_OIL_TANK] = {
        "Oil Tank", "Industrial oil storage tank", PRIM_OIL_TANK,
        {AlgorithmParameter("Radius", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimOilTank
    };

    algorithms[PRIM_GEOSPHERE] = {
        "GeoSphere", "Geodesic sphere with triangular faces", PRIM_GEOSPHERE,
        {AlgorithmParameter("Radius", 1.0f, 0.1f, 5.0f), AlgorithmParameter("Subdivisions", 2.0f, 0.0f, 4.0f, 1.0f)},
        generatePrimGeosphere
    };

    algorithms[PRIM_TUBE] = {
        "Tube", "Cylindrical tube with thickness", PRIM_TUBE,
        {AlgorithmParameter("Radius", 1.0f, 0.1f, 3.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Thickness", 0.2f, 0.05f, 0.5f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimTube
    };

    algorithms[PRIM_PYRAMID] = {
        "Pyramid", "Pyramid with square base", PRIM_PYRAMID,
        {AlgorithmParameter("Base Size", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Subdivisions", 1.0f, 1.0f, 8.0f, 1.0f)},
        generatePrimPyramid
    };

    algorithms[PRIM_HEDRA] = {
        "Hedra", "General polyhedron", PRIM_HEDRA,
        {AlgorithmParameter("Radius", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Sides", 6.0f, 3.0f, 12.0f, 1.0f), AlgorithmParameter("Subdivision", 1.0f, 1.0f, 4.0f, 1.0f)},
        generatePrimHedra
    };

    algorithms[PRIM_TORUS_KNOT] = {
        "Torus Knot", "Parametric torus knot", PRIM_TORUS_KNOT,
        {AlgorithmParameter("P", 2.0f, 1.0f, 8.0f, 1.0f), AlgorithmParameter("Q", 3.0f, 1.0f, 8.0f, 1.0f), AlgorithmParameter("Tube Radius", 0.3f, 0.1f, 0.8f), AlgorithmParameter("Segments", 128.0f, 32.0f, 256.0f, 16.0f)},
        generatePrimTorusKnot
    };

    algorithms[PRIM_CAPSULE] = {
        "Capsule", "Capsule/pill shape", PRIM_CAPSULE,
        {AlgorithmParameter("Radius", 0.5f, 0.1f, 2.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 32.0f, 8.0f, 64.0f)},
        generatePrimCapsule
    };

    algorithms[PRIM_SPINDLE] = {
        "Spindle", "Spindle/double cone shape", PRIM_SPINDLE,
        {AlgorithmParameter("Radius", 1.0f, 0.3f, 3.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimSpindle
    };

    algorithms[PRIM_L_EXT] = {
        "L-Ext", "L-shaped extrusion", PRIM_L_EXT,
        {AlgorithmParameter("Width", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Thickness", 0.3f, 0.1f, 1.0f), AlgorithmParameter("Depth", 1.0f, 0.3f, 3.0f)},
        generatePrimLExt
    };

    algorithms[PRIM_C_EXT] = {
        "C-Ext", "C-shaped extrusion", PRIM_C_EXT,
        {AlgorithmParameter("Width", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Thickness", 0.3f, 0.1f, 1.0f), AlgorithmParameter("Depth", 1.0f, 0.3f, 3.0f)},
        generatePrimCExt
    };

    algorithms[PRIM_GENGON] = {
        "Gengon", "General polygon with fillet", PRIM_GENGON,
        {AlgorithmParameter("Radius", 1.0f, 0.3f, 3.0f), AlgorithmParameter("Sides", 6.0f, 3.0f, 16.0f, 1.0f), AlgorithmParameter("Fillet", 0.2f, 0.0f, 0.5f), AlgorithmParameter("Height", 1.0f, 0.3f, 3.0f)},
        generatePrimGengon
    };

    algorithms[PRIM_RINGWAVE] = {
        "RingWave", "Ring with wave pattern", PRIM_RINGWAVE,
        {AlgorithmParameter("Radius", 1.5f, 0.5f, 3.0f), AlgorithmParameter("Thickness", 0.3f, 0.1f, 0.8f), AlgorithmParameter("Wave Amplitude", 0.2f, 0.0f, 0.5f), AlgorithmParameter("Wave Frequency", 8.0f, 2.0f, 20.0f, 1.0f), AlgorithmParameter("Segments", 64.0f, 16.0f, 128.0f)},
        generatePrimRingWave
    };

    algorithms[PRIM_HOSE] = {
        "Hose", "Flexible hose/tube", PRIM_HOSE,
        {AlgorithmParameter("Radius", 0.3f, 0.1f, 1.0f), AlgorithmParameter("Length", 3.0f, 1.0f, 8.0f), AlgorithmParameter("Bend Angle", 45.0f, 0.0f, 180.0f), AlgorithmParameter("Segments", 40.0f, 8.0f, 64.0f)},
        generatePrimHose
    };

    algorithms[PRIM_PRISM] = {
        "Prism", "Polygonal prism", PRIM_PRISM,
        {AlgorithmParameter("Radius", 1.0f, 0.3f, 3.0f), AlgorithmParameter("Sides", 6.0f, 3.0f, 16.0f, 1.0f), AlgorithmParameter("Height", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Height Segments", 1.0f, 1.0f, 8.0f, 1.0f)},
        generatePrimPrism
    };

    // PLATONIC SOLIDS
    algorithms[PLATO_TETRAHEDRON] = {
        "Tetrahedron", "Regular tetrahedron", PLATO_TETRAHEDRON,
        {AlgorithmParameter("Size", 1.0f, 0.5f, 3.0f)},
        generatePlatoTetrahedron
    };

    algorithms[PLATO_OCTAHEDRON] = {
        "Octahedron", "Regular octahedron", PLATO_OCTAHEDRON,
        {AlgorithmParameter("Size", 1.0f, 0.5f, 3.0f)},
        generatePlatoOctahedron
    };

    algorithms[PLATO_ICOSAHEDRON] = {
        "Icosahedron", "Regular icosahedron", PLATO_ICOSAHEDRON,
        {AlgorithmParameter("Size", 1.0f, 0.5f, 3.0f)},
        generatePlatoIcosahedron
    };

    algorithms[PLATO_DODECAHEDRON] = {
        "Dodecahedron", "Regular dodecahedron", PLATO_DODECAHEDRON,
        {AlgorithmParameter("Size", 1.0f, 0.5f, 3.0f)},
        generatePlatoDodecahedron
    };

    // PARAMETRIC SURFACES (additional)
    algorithms[PARAM_LISSAJOUS] = {
        "Lissajous", "3D Lissajous curve", PARAM_LISSAJOUS,
        {AlgorithmParameter("A", 3.0f, 1.0f, 10.0f), AlgorithmParameter("B", 2.0f, 1.0f, 10.0f), AlgorithmParameter("C", 5.0f, 1.0f, 10.0f), AlgorithmParameter("Segments", 200.0f, 50.0f, 500.0f)},
        generateParamLissajous
    };

    algorithms[PARAM_ROSE] = {
        "Rose", "3D rose curve", PARAM_ROSE,
        {AlgorithmParameter("N", 5.0f, 2.0f, 12.0f), AlgorithmParameter("D", 3.0f, 1.0f, 12.0f), AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Segments", 200.0f, 50.0f, 500.0f)},
        generateParamRose
    };

    algorithms[PARAM_HELIX] = {
        "Helix", "3D helical curve", PARAM_HELIX,
        {AlgorithmParameter("Radius", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Height", 3.0f, 1.0f, 10.0f), AlgorithmParameter("Turns", 5.0f, 1.0f, 20.0f), AlgorithmParameter("Segments", 100.0f, 20.0f, 300.0f)},
        generateParamHelix
    };

    // TOPOLOGY (additional)
    algorithms[TOPO_FIGURE_EIGHT_KNOT] = {
        "Figure-Eight Knot", "Classic figure-eight knot", TOPO_FIGURE_EIGHT_KNOT,
        {AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Tube Radius", 0.2f, 0.1f, 0.5f), AlgorithmParameter("Segments", 128.0f, 64.0f, 256.0f)},
        generateTopoFigureEightKnot
    };

    algorithms[TOPO_HEART] = {
        "Heart", "3D heart curve", TOPO_HEART,
        {AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Segments", 64.0f, 32.0f, 128.0f)},
        generateTopoHeart
    };

    // MINIMAL SURFACES (additional)
    algorithms[MIN_NEOVIUS] = {
        "Neovius (Surfels)", "Neovius minimal surface", MIN_NEOVIUS,
        {AlgorithmParameter("Scale", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Thickness", 0.0f, -0.5f, 0.5f), AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f)},
        generateMinNeovius
    };

    algorithms[MIN_HELICOID] = {
        "Helicoid", "Helicoid minimal surface", MIN_HELICOID,
        {AlgorithmParameter("Radius", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Height", 3.0f, 1.0f, 10.0f), AlgorithmParameter("Turns", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Segments", 64.0f, 32.0f, 128.0f)},
        generateMinHelicoid
    };

    algorithms[MIN_CATENOID] = {
        "Catenoid", "Catenoid minimal surface", MIN_CATENOID,
        {AlgorithmParameter("Radius", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Height", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Segments", 64.0f, 32.0f, 128.0f)},
        generateMinCatenoid
    };

    algorithms[MIN_ENNEPER] = {
        "Enneper", "Enneper minimal surface", MIN_ENNEPER,
        {AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f)},
        generateMinEnneper
    };

    // MINIMAL SURFACES - Marching Cubes variants
    algorithms[MIN_GYROID_MC] = {
        "Gyroid (Marching Cubes)", "Gyroid surface using marching cubes", MIN_GYROID_MC,
        {AlgorithmParameter("Scale", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Thickness", 0.0f, -0.5f, 0.5f), AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f)},
        generateGyroidSurfaceMC
    };

    algorithms[MIN_SCHWARZ_P_MC] = {
        "Schwarz P (Marching Cubes)", "Schwarz P surface using marching cubes", MIN_SCHWARZ_P_MC,
        {AlgorithmParameter("Scale", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Thickness", 0.0f, -0.5f, 0.5f), AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f)},
        generateSchwarzPSurfaceMC
    };

    algorithms[MIN_NEOVIUS_MC] = {
        "Neovius (Marching Cubes)", "Neovius surface using marching cubes", MIN_NEOVIUS_MC,
        {AlgorithmParameter("Scale", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Thickness", 0.0f, -0.5f, 0.5f), AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f)},
        generateMinNeoviusMC
    };

    // POINT CLOUD (111-125)
    // Mesh Method: 0=Points, 1=Convex Hull, 2=Delaunay, 3=Alpha Shapes, 4=Ball Pivot, 5=Power Crust, 6=Poisson, 7=Marching Cubes, 8=Splats, 9=Metaballs
    algorithms[PC_RANDOM_SCATTER] = {
        "Random Scatter", "Random point distribution", PC_RANDOM_SCATTER,
        {AlgorithmParameter("Count", 1000.0f, 100.0f, 10000.0f, 100.0f), AlgorithmParameter("Radius", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCRandomScatter
    };

    algorithms[PC_GRID_SAMPLE] = {
        "Grid Sample", "Uniform grid sampling", PC_GRID_SAMPLE,
        {AlgorithmParameter("Grid X", 10.0f, 5.0f, 50.0f, 1.0f), AlgorithmParameter("Grid Y", 10.0f, 5.0f, 50.0f, 1.0f), AlgorithmParameter("Grid Z", 10.0f, 5.0f, 50.0f, 1.0f), AlgorithmParameter("Spacing", 0.2f, 0.05f, 1.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCGridSample
    };

    algorithms[PC_POISSON_DISK] = {
        "Poisson Disk", "Poisson disk sampling", PC_POISSON_DISK,
        {AlgorithmParameter("Count", 500.0f, 50.0f, 5000.0f, 50.0f), AlgorithmParameter("Min Distance", 0.2f, 0.05f, 1.0f), AlgorithmParameter("Radius", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCPoissonDisk
    };

    algorithms[PC_BLUE_NOISE] = {
        "Blue Noise", "Blue noise sampling", PC_BLUE_NOISE,
        {AlgorithmParameter("Count", 1000.0f, 100.0f, 10000.0f, 100.0f), AlgorithmParameter("Radius", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCBlueNoise
    };

    algorithms[PC_SURFACE_SAMPLE] = {
        "Surface Sample", "Sample points on surface", PC_SURFACE_SAMPLE,
        {AlgorithmParameter("Count", 1000.0f, 100.0f, 10000.0f, 100.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCSurfaceSample
    };

    algorithms[PC_VOLUME_SAMPLE] = {
        "Volume Sample", "Sample points in volume", PC_VOLUME_SAMPLE,
        {AlgorithmParameter("Count", 1000.0f, 100.0f, 10000.0f, 100.0f), AlgorithmParameter("Radius", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCVolumeSample
    };

    algorithms[PC_FIBONACCI_SPHERE] = {
        "Fibonacci Sphere", "Fibonacci spiral distribution", PC_FIBONACCI_SPHERE,
        {AlgorithmParameter("Count", 500.0f, 50.0f, 5000.0f, 50.0f), AlgorithmParameter("Radius", 1.0f, 0.3f, 3.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCFibonacciSphere
    };

    algorithms[PC_HALTON_SEQUENCE] = {
        "Halton Sequence", "Low-discrepancy Halton sampling", PC_HALTON_SEQUENCE,
        {AlgorithmParameter("Count", 1000.0f, 100.0f, 10000.0f, 100.0f), AlgorithmParameter("Base 1", 2.0f, 2.0f, 7.0f, 1.0f), AlgorithmParameter("Base 2", 3.0f, 2.0f, 7.0f, 1.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCHaltonSequence
    };

    algorithms[PC_SOBOL_SEQUENCE] = {
        "Sobol Sequence", "Quasi-random Sobol sampling", PC_SOBOL_SEQUENCE,
        {AlgorithmParameter("Count", 1000.0f, 100.0f, 10000.0f, 100.0f), AlgorithmParameter("Scale", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCSobolSequence
    };

    algorithms[PC_STRATIFIED_SAMPLE] = {
        "Stratified Sample", "Stratified random sampling", PC_STRATIFIED_SAMPLE,
        {AlgorithmParameter("Strata X", 10.0f, 3.0f, 20.0f, 1.0f), AlgorithmParameter("Strata Y", 10.0f, 3.0f, 20.0f, 1.0f), AlgorithmParameter("Strata Z", 10.0f, 3.0f, 20.0f, 1.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCStratifiedSample
    };

    algorithms[PC_SPHERE_PACKING] = {
        "Sphere Packing", "Dense sphere packing", PC_SPHERE_PACKING,
        {AlgorithmParameter("Count", 100.0f, 10.0f, 500.0f, 10.0f), AlgorithmParameter("Sphere Radius", 0.1f, 0.05f, 0.3f), AlgorithmParameter("Domain Size", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCSpherePacking
    };

    algorithms[PC_VOXEL_DOWNSAMPLE] = {
        "Voxel Downsample", "Voxel-based downsampling", PC_VOXEL_DOWNSAMPLE,
        {AlgorithmParameter("Voxel Size", 0.2f, 0.05f, 1.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCVoxelDownsample
    };

    algorithms[PC_NORMAL_ESTIMATE] = {
        "Normal Estimate", "Estimate point normals", PC_NORMAL_ESTIMATE,
        {AlgorithmParameter("K Neighbors", 10.0f, 3.0f, 30.0f, 1.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCNormalEstimate
    };

    algorithms[PC_OUTLIER_REMOVAL] = {
        "Outlier Removal", "Remove statistical outliers", PC_OUTLIER_REMOVAL,
        {AlgorithmParameter("K Neighbors", 10.0f, 3.0f, 30.0f, 1.0f), AlgorithmParameter("Std Dev", 2.0f, 0.5f, 5.0f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCOutlierRemoval
    };

    algorithms[PC_ICP_ALIGN] = {
        "ICP Align", "Iterative Closest Point alignment", PC_ICP_ALIGN,
        {AlgorithmParameter("Iterations", 20.0f, 5.0f, 100.0f, 5.0f), AlgorithmParameter("Tolerance", 0.001f, 0.0001f, 0.01f, 0.001f), AlgorithmParameter("Mesh Method", 1.08f, 0.0f, 9.0f, 1.0f), AlgorithmParameter("Method Param", 0.5f, 0.1f, 2.0f)},
        generatePCICPAlign
    };

    // ORGANIC (additional)
    algorithms[ORG_SHELL_CONCH] = {
        "Conch Shell", "Conical shell", ORG_SHELL_CONCH,
        {AlgorithmParameter("Growth Rate", 0.15f, 0.05f, 0.5f), AlgorithmParameter("Turns", 4.0f, 2.0f, 10.0f), AlgorithmParameter("Radius", 0.8f, 0.3f, 2.0f), AlgorithmParameter("Segments", 64.0f, 32.0f, 128.0f)},
        generateOrgShellConch
    };

    algorithms[ORG_CRYSTAL] = {
        "Crystal", "Geometric crystal", ORG_CRYSTAL,
        {AlgorithmParameter("Size", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Facets", 8.0f, 4.0f, 16.0f), AlgorithmParameter("Complexity", 2.0f, 1.0f, 5.0f)},
        generateOrgCrystal
    };

    algorithms[ORG_TENTACLE] = {
        "Tentacle", "Organic tentacle", ORG_TENTACLE,
        {AlgorithmParameter("Length", 3.0f, 1.0f, 10.0f), AlgorithmParameter("Radius", 0.3f, 0.1f, 1.0f), AlgorithmParameter("Segments", 40.0f, 16.0f, 64.0f), AlgorithmParameter("Twist", 2.0f, 0.0f, 10.0f)},
        generateOrgTentacle
    };

    // FRACTALS (additional)
    algorithms[FRAC_MOUNTAIN] = {
        "Fractal Mountain", "Terrain using diamond-square", FRAC_MOUNTAIN,
        {AlgorithmParameter("Size", 8.0f, 4.0f, 16.0f), AlgorithmParameter("Height", 3.0f, 1.0f, 10.0f), AlgorithmParameter("Roughness", 0.5f, 0.1f, 1.0f), AlgorithmParameter("Iterations", 5.0f, 2.0f, 8.0f)},
        generateFracMountain
    };

    algorithms[FRAC_FERN] = {
        "Barnsley Fern", "Iterated function system fern", FRAC_FERN,
        {AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Iterations", 10000.0f, 1000.0f, 50000.0f), AlgorithmParameter("Thickness", 0.05f, 0.01f, 0.2f)},
        generateFracFern
    };

    algorithms[FRAC_MANDELBROT_2D] = {
        "Mandelbrot 2D", "2D Mandelbrot set as height field visualization", FRAC_MANDELBROT_2D,
        {AlgorithmParameter("Resolution", 64.0f, 32.0f, 128.0f), AlgorithmParameter("Zoom", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Height Scale", 0.5f, 0.1f, 2.0f)},
        generateFracMandelbrot2D
    };

    algorithms[FRAC_MANDELBROT_3D] = {
        "Mandelbrot 3D", "3D Mandelbrot set fractal visualization", FRAC_MANDELBROT_3D,
        {AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f), AlgorithmParameter("Size", 2.0f, 1.0f, 4.0f), AlgorithmParameter("Iterations", 8.0f, 4.0f, 16.0f)},
        generateFracMandelbrot3D
    };

    algorithms[FRAC_MENGER_SPONGE] = {
        "Menger Sponge", "3D Cantor set fractal", FRAC_MENGER_SPONGE,
        {AlgorithmParameter("Size", 3.0f, 1.0f, 5.0f), AlgorithmParameter("Iterations", 3.0f, 1.0f, 5.0f)},
        generateFracMengerSponge
    };

    algorithms[FRAC_JULIA_3D] = {
        "Julia Set 3D", "3D Julia set", FRAC_JULIA_3D,
        {AlgorithmParameter("C Real", 0.285f, -1.0f, 1.0f), AlgorithmParameter("C Imag", 0.01f, -1.0f, 1.0f), AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f), AlgorithmParameter("Iterations", 10.0f, 5.0f, 20.0f)},
        generateFracJulia3D
    };

    algorithms[FRAC_JULIA_2D] = {
        "Julia Set 2D", "2D Julia set complex plane visualization", FRAC_JULIA_2D,
        {AlgorithmParameter("C Real", -0.4f, -1.0f, 1.0f), AlgorithmParameter("C Imag", 0.6f, -1.0f, 1.0f), AlgorithmParameter("Resolution", 64.0f, 32.0f, 128.0f), AlgorithmParameter("Height Scale", 0.5f, 0.1f, 2.0f)},
        generateFracJulia2D
    };

    algorithms[FRAC_SIERPINSKI] = {
        "Sierpinski", "Sierpinski triangle or pyramid fractal", FRAC_SIERPINSKI,
        {AlgorithmParameter("Iterations", 4.0f, 1.0f, 6.0f), AlgorithmParameter("Size", 2.0f, 1.0f, 4.0f), AlgorithmParameter("3D Mode", 0.0f, 0.0f, 1.0f)},
        generateFracSierpinski
    };

    algorithms[FRAC_KOCH_SNOWFLAKE] = {
        "Koch Snowflake", "Koch snowflake curve fractal", FRAC_KOCH_SNOWFLAKE,
        {AlgorithmParameter("Iterations", 4.0f, 1.0f, 6.0f), AlgorithmParameter("Size", 2.0f, 1.0f, 4.0f), AlgorithmParameter("Thickness", 0.1f, 0.05f, 0.3f)},
        generateFracKochSnowflake
    };

    algorithms[FRAC_DRAGON_CURVE] = {
        "Dragon Curve", "Heighway dragon fractal", FRAC_DRAGON_CURVE,
        {AlgorithmParameter("Iterations", 10.0f, 5.0f, 15.0f), AlgorithmParameter("Scale", 1.0f, 0.5f, 3.0f), AlgorithmParameter("Thickness", 0.1f, 0.05f, 0.3f)},
        generateFracDragonCurve
    };

    // STRUCTURAL (additional)
    algorithms[STRUCT_DIAMOND_LATTICE] = {
        "Diamond Lattice", "Diamond crystal lattice", STRUCT_DIAMOND_LATTICE,
        {AlgorithmParameter("Size", 3.0f, 1.0f, 6.0f), AlgorithmParameter("Cell Count", 3.0f, 1.0f, 5.0f), AlgorithmParameter("Thickness", 0.1f, 0.05f, 0.3f)},
        generateStructDiamondLattice
    };

    algorithms[STRUCT_PIPE_NETWORK] = {
        "Pipe Network", "Branching pipe system", STRUCT_PIPE_NETWORK,
        {AlgorithmParameter("Branches", 8.0f, 4.0f, 16.0f), AlgorithmParameter("Length", 2.0f, 1.0f, 5.0f), AlgorithmParameter("Radius", 0.2f, 0.1f, 0.5f), AlgorithmParameter("Levels", 3.0f, 1.0f, 5.0f)},
        generateStructPipeNetwork
    };

    algorithms[STRUCT_LATTICE] = {
        "Lattice", "Regular 3D lattice", STRUCT_LATTICE,
        {AlgorithmParameter("Size", 5.0f, 2.0f, 10.0f), AlgorithmParameter("Divisions", 5.0f, 2.0f, 10.0f), AlgorithmParameter("Thickness", 0.1f, 0.05f, 0.3f)},
        generateStructLattice
    };

    // COMPUTATIONAL (additional)
    algorithms[COMP_VORONOI_SURFACE] = {
        "Voronoi Surface", "Voronoi tessellation creating cellular patterns", COMP_VORONOI_SURFACE,
        {AlgorithmParameter("Point Count", 20.0f, 10.0f, 50.0f), AlgorithmParameter("Size", 5.0f, 2.0f, 10.0f), AlgorithmParameter("Height", 1.0f, 0.5f, 3.0f)},
        generateCompVoronoiSurface
    };

    algorithms[COMP_DELAUNAY_SURFACE] = {
        "Delaunay Surface", "Delaunay triangulation", COMP_DELAUNAY_SURFACE,
        {AlgorithmParameter("Point Count", 20.0f, 10.0f, 50.0f), AlgorithmParameter("Size", 5.0f, 2.0f, 10.0f)},
        generateCompDelaunaySurface
    };

    algorithms[COMP_VORONOI_DIAGRAM] = {
        "Voronoi Diagram", "2D Voronoi diagram with cell visualization", COMP_VORONOI_DIAGRAM,
        {AlgorithmParameter("Point Count", 20.0f, 10.0f, 50.0f), AlgorithmParameter("Size", 5.0f, 2.0f, 10.0f), AlgorithmParameter("Cell Height", 1.0f, 0.5f, 2.0f)},
        generateCompVoronoiDiagram
    };

    algorithms[COMP_DELAUNAY_TRIANGULATION] = {
        "Delaunay Triangulation", "2D Delaunay triangulation mesh", COMP_DELAUNAY_TRIANGULATION,
        {AlgorithmParameter("Point Count", 30.0f, 10.0f, 100.0f), AlgorithmParameter("Size", 5.0f, 2.0f, 10.0f)},
        generateCompDelaunayTriangulation
    };

    algorithms[COMP_PERLIN_NOISE_SURFACE] = {
        "Perlin Noise Surface", "Terrain-like surface using Perlin noise", COMP_PERLIN_NOISE_SURFACE,
        {AlgorithmParameter("Resolution", 64.0f, 32.0f, 128.0f), AlgorithmParameter("Scale", 0.1f, 0.05f, 0.5f), AlgorithmParameter("Height", 5.0f, 1.0f, 10.0f)},
        generateCompPerlinNoiseSurface
    };

    algorithms[COMP_NEURAL_NETWORK] = {
        "Neural Network", "Neural network-generated parametric surface", COMP_NEURAL_NETWORK,
        {AlgorithmParameter("Resolution", 32.0f, 16.0f, 64.0f), AlgorithmParameter("Scale", 3.0f, 1.0f, 5.0f)},
        generateCompNeuralNetwork
    };

    algorithms[COMP_CPPN_HYPERNEAT] = {
        "CPPN-HyperNEAT", "Compositional pattern-producing network mesh generation", COMP_CPPN_HYPERNEAT,
        {AlgorithmParameter("Resolution", 16.0f, 8.0f, 32.0f)},
        generateCompCPPNHyperNEAT
    };

    algorithms[COMP_EVOLUTIONARY_SHAPE] = {
        "Evolutionary Shape", "Evolutionary algorithm for shape optimization", COMP_EVOLUTIONARY_SHAPE,
        {AlgorithmParameter("Generations", 20.0f, 5.0f, 50.0f), AlgorithmParameter("Population", 10.0f, 5.0f, 20.0f), AlgorithmParameter("Complexity", 5.0f, 1.0f, 10.0f)},
        generateCompEvolutionaryShape
    };

    // Additional modifiers
    algorithms[MOD_VOXELIZE] = {
        "Voxelize", "Convert mesh to voxel representation", MOD_VOXELIZE,
        {AlgorithmParameter("Grid Size", 32.0f, 16.0f, 64.0f), AlgorithmParameter("Voxel Size", 1.0f, 0.5f, 2.0f)},
        generateModVoxelize
    };

    algorithms[MOD_LAPLACIAN_RELAX] = {
        "Laplacian Relax", "Mesh smoothing using Laplacian relaxation", MOD_LAPLACIAN_RELAX,
        {AlgorithmParameter("Iterations", 5.0f, 1.0f, 20.0f), AlgorithmParameter("Factor", 0.5f, 0.1f, 1.0f)},
        generateModLaplacianRelax
    };

    algorithms[MOD_CMAES_OPTIMIZE] = {
        "CMA-ES Optimize", "Optimize mesh shape using CMA-ES evolutionary strategy", MOD_CMAES_OPTIMIZE,
        {AlgorithmParameter("Generations", 20.0f, 5.0f, 100.0f), AlgorithmParameter("Population", 10.0f, 5.0f, 30.0f), AlgorithmParameter("Target", 1.0f, 0.5f, 2.0f)},
        generateModCMAESOptimize
    };

    algorithms[MOD_NSGAII_OPTIMIZE] = {
        "NSGA-II Multi-Objective", "Multi-objective optimization using NSGA-II", MOD_NSGAII_OPTIMIZE,
        {AlgorithmParameter("Generations", 30.0f, 10.0f, 100.0f), AlgorithmParameter("Population", 20.0f, 10.0f, 50.0f), AlgorithmParameter("Complexity", 5.0f, 1.0f, 10.0f)},
        generateModNSGAIIOptimize
    };

    algorithms[MOD_RBF_OPTIMIZE] = {
        "RBF Surrogate", "Surrogate-based optimization using radial basis functions", MOD_RBF_OPTIMIZE,
        {AlgorithmParameter("Evaluations", 50.0f, 20.0f, 200.0f), AlgorithmParameter("Smoothness", 1.0f, 0.5f, 2.0f)},
        generateModRBFOptimize
    };
}

Mesh AlgorithmLibrary::generate(AlgorithmType type, const std::vector<AlgorithmParameter>& params) {
    auto it = algorithms.find(type);
    if (it != algorithms.end()) {
        return it->second.function(params);
    }
    return Mesh();
}

std::vector<std::string> AlgorithmLibrary::getAllAlgorithmNames() const {
    std::vector<std::string> names;
    for (const auto& pair : algorithms) {
        names.push_back(pair.second.name);
    }
    return names;
}

std::vector<AlgorithmLibrary::AlgorithmType> AlgorithmLibrary::getAllRegisteredTypes() const {
    std::vector<AlgorithmType> types;
    for (const auto& pair : algorithms) {
        types.push_back(pair.first);
    }
    return types;
}

// Implementation of individual algorithms

Mesh AlgorithmLibrary::generateSpiralSphere(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    int spirals = (int)params[1].value;
    float turns = params[2].value;
    int resolution = (int)params[3].value;

    Mesh mesh = MeshGenerator::createSphere(radius, resolution, resolution / 2);

    // Add spiral displacement
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float angle = std::atan2(v.position.z, v.position.x);
        float height = v.position.y / radius;
        float spiral = std::sin(spirals * angle + turns * height * M_PI);
        v.position = v.position * (1.0f + 0.1f * spiral);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateTwistedTorus(const std::vector<AlgorithmParameter>& params) {
    float majorRadius = params[0].value;
    float minorRadius = params[1].value;
    float twistAmount = params[2].value;
    int segments = (int)params[3].value;

    Mesh mesh;

    for (int i = 0; i <= segments; i++) {
        float u = 2.0f * M_PI * i / segments;
        for (int j = 0; j <= segments / 2; j++) {
            float v = 2.0f * M_PI * j / (segments / 2);

            float twist = twistAmount * u;
            float x = (majorRadius + minorRadius * std::cos(v + twist)) * std::cos(u);
            float y = minorRadius * std::sin(v + twist);
            float z = (majorRadius + minorRadius * std::cos(v + twist)) * std::sin(u);

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    int cols = segments / 2 + 1;
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < segments / 2; j++) {
            int idx = i * cols + j;
            int next = idx + cols;
            mesh.addTriangle(idx, next, idx + 1);
            mesh.addTriangle(idx + 1, next, next + 1);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateGeodesicDome(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    int subdivisions = (int)params[1].value;

    Mesh mesh = MeshGenerator::createIcosahedron(radius);
    MeshGenerator::subdivide(mesh, subdivisions);

    // Project to sphere
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        mesh.getVertex(i).position = mesh.getVertex(i).position.normalized() * radius;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateParametricWave(const std::vector<AlgorithmParameter>& params) {
    float amplitude = params[0].value;
    float freqX = params[1].value;
    float freqY = params[2].value;
    int resolution = (int)params[3].value;

    Mesh mesh;
    float size = 5.0f;

    for (int y = 0; y <= resolution; y++) {
        for (int x = 0; x <= resolution; x++) {
            float px = (float)x / resolution * size - size / 2;
            float pz = (float)y / resolution * size - size / 2;
            float py = amplitude * std::sin(freqX * px) * std::cos(freqY * pz);

            mesh.addVertex(Vertex(Vec3(px, py, pz)));
        }
    }

    for (int y = 0; y < resolution; y++) {
        for (int x = 0; x < resolution; x++) {
            int i0 = y * (resolution + 1) + x;
            int i1 = i0 + 1;
            int i2 = i0 + resolution + 1;
            int i3 = i2 + 1;

            mesh.addTriangle(i0, i2, i1);
            mesh.addTriangle(i1, i2, i3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateShellStructure(const std::vector<AlgorithmParameter>& params) {
    float growthRate = params[0].value;
    float turns = params[1].value;
    float radius = params[2].value;
    int segments = (int)params[3].value;

    Mesh mesh;

    for (int i = 0; i <= segments; i++) {
        float t = (float)i / segments * turns * 2 * M_PI;
        float r = radius * std::exp(growthRate * t);

        for (int j = 0; j <= 16; j++) {
            float angle = 2.0f * M_PI * j / 16;
            float innerR = r * 0.3f;
            float x = (r + innerR * std::cos(angle)) * std::cos(t);
            float y = (r + innerR * std::cos(angle)) * std::sin(t);
            float z = innerR * std::sin(angle);

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < 16; j++) {
            int idx = i * 17 + j;
            int next = idx + 17;
            mesh.addTriangle(idx, next, idx + 1);
            mesh.addTriangle(idx + 1, next, next + 1);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateLathe(const std::vector<AlgorithmParameter>& params) {
    int segments = (int)params[0].value;
    float profileScale = params[1].value;

    // Create a simple profile curve
    std::vector<Vec3> profile = {
        Vec3(0.5f * profileScale, 0, 0),
        Vec3(1.0f * profileScale, 0.3f, 0),
        Vec3(0.8f * profileScale, 0.6f, 0),
        Vec3(0.5f * profileScale, 1.0f, 0)
    };

    return MeshGenerator::lathe(profile, segments);
}

Mesh AlgorithmLibrary::generateLoft(const std::vector<AlgorithmParameter>& params) {
    int segments = (int)params[0].value;
    float blendFactor = params[1].value;

    std::vector<std::vector<Vec3>> profiles;

    // Profile 1: Circle at y=0
    std::vector<Vec3> p1;
    for (int i = 0; i < segments; i++) {
        float angle = i * 2.0f * M_PI / segments;
        p1.push_back(Vec3(std::cos(angle), 0, std::sin(angle)));
    }

    // Profile 2: Scaled circle at y=2, size influenced by blend factor
    std::vector<Vec3> p2;
    float scale = 0.5f + blendFactor * 0.5f;
    for (int i = 0; i < segments; i++) {
        float angle = i * 2.0f * M_PI / segments;
        p2.push_back(Vec3(scale * std::cos(angle), 2, scale * std::sin(angle)));
    }

    profiles.push_back(p1);
    profiles.push_back(p2);

    return MeshGenerator::loft(profiles);
}

Mesh AlgorithmLibrary::generateVoronoiLattice(const std::vector<AlgorithmParameter>& params) {
    int cellCount = (int)params[0].value;
    float size = params[1].value;
    float thickness = params[2].value;

    Mesh mesh;

    std::mt19937 gen(42); // Fixed seed for consistency
    std::uniform_real_distribution<float> dis(-size/2, size/2);

    std::vector<Vec3> points;
    for (int i = 0; i < cellCount; i++) {
        points.push_back(Vec3(dis(gen), dis(gen), dis(gen)));
    }

    // Create connections between nearby points
    for (size_t i = 0; i < points.size(); i++) {
        for (size_t j = i + 1; j < points.size(); j++) {
            Vec3 dir = points[j] - points[i];
            float dist = dir.length();

            if (dist < size * 0.5f && dist > 0.001f) {
                dir.normalize();

                Mesh cylinder = MeshGenerator::createCylinder(thickness, dist, 8);

                // Calculate rotation to align cylinder with direction
                Vec3 up = dir;
                float tilt = std::acos(up.y);
                float angle = std::atan2(up.x, up.z);

                cylinder.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(tilt, 0, angle));
                cylinder.transform(points[i], Vec3(1, 1, 1), Vec3(0, 0, 0));

                mesh.merge(cylinder);

                // Add spheres at connection points
                Mesh sphere = MeshGenerator::createSphere(thickness * 1.5f, 6, 4);
                sphere.transform(points[i], Vec3(1, 1, 1), Vec3(0, 0, 0));
                mesh.merge(sphere);
            }
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateHexagonalGrid(const std::vector<AlgorithmParameter>& params) {
    int rings = (int)params[0].value;
    float radius = params[1].value;
    float height = params[2].value;
    float gap = params[3].value;

    Mesh mesh;

    for (int q = -rings; q <= rings; q++) {
        int r1 = std::max(-rings, -q - rings);
        int r2 = std::min(rings, -q + rings);

        for (int r = r1; r <= r2; r++) {
            float x = radius * (std::sqrt(3.0f) * q + std::sqrt(3.0f) / 2.0f * r);
            float z = radius * (3.0f / 2.0f * r);

            // Create hexagon
            Mesh hex;
            for (int i = 0; i < 6; i++) {
                float angle = M_PI / 3.0f * i;
                float hx = (radius - gap) * std::cos(angle);
                float hz = (radius - gap) * std::sin(angle);
                hex.addVertex(Vertex(Vec3(hx, 0, hz)));
                hex.addVertex(Vertex(Vec3(hx, height, hz)));
            }

            // Add faces
            for (int i = 0; i < 6; i++) {
                int next = (i + 1) % 6;
                int b0 = i * 2, b1 = next * 2;
                int t0 = b0 + 1, t1 = b1 + 1;
                hex.addTriangle(b0, b1, t0);
                hex.addTriangle(t0, b1, t1);
            }

            hex.transform(Vec3(x, 0, z), Vec3(1, 1, 1), Vec3(0, 0, 0));
            mesh.merge(hex);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateBuildingMassing(const std::vector<AlgorithmParameter>& params) {
    float width = params[0].value;
    float depth = params[1].value;
    float height = params[2].value;
    int floors = (int)params[3].value;
    float setback = params[4].value;

    Mesh mesh;
    float floorHeight = height / floors;

    for (int f = 0; f < floors; f++) {
        float factor = 1.0f - (1.0f - setback) * f / floors;
        float w = width * factor;
        float d = depth * factor;
        float h = floorHeight;
        float y = f * floorHeight;

        Mesh box = MeshGenerator::createCube(1.0f);
        box.transform(Vec3(0, y + h/2, 0), Vec3(w, h, d), Vec3(0, 0, 0));
        mesh.merge(box);
    }

    return mesh;
}

Mesh AlgorithmLibrary::generateFacadePattern(const std::vector<AlgorithmParameter>& params) {
    float width = params[0].value;
    float height = params[1].value;
    float panelWidth = params[2].value;
    float panelHeight = params[3].value;
    float depthVar = params[4].value;

    Mesh mesh;

    int panelsX = (int)(width / panelWidth);
    int panelsY = (int)(height / panelHeight);

    std::mt19937 gen(42);
    std::uniform_real_distribution<double> dis(0, depthVar);

    for (int y = 0; y < panelsY; y++) {
        for (int x = 0; x < panelsX; x++) {
            float px = x * panelWidth - width / 2;
            float py = y * panelHeight;
            float depth = dis(gen);

            Mesh panel = MeshGenerator::createCube(1.0f);
            panel.transform(
                Vec3(px + panelWidth/2, py + panelHeight/2, depth/2),
                Vec3(panelWidth * 0.9f, panelHeight * 0.9f, depth),
                Vec3(0, 0, 0)
            );
            mesh.merge(panel);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateStructuralFrame(const std::vector<AlgorithmParameter>& params) {
    float width = params[0].value;
    float height = params[1].value;
    float depth = params[2].value;
    int divisions = (int)params[3].value;
    float memberSize = params[4].value;

    Mesh mesh;

    // Vertical members
    for (int x = 0; x <= divisions; x++) {
        for (int z = 0; z <= divisions; z++) {
            float px = (float)x / divisions * width - width / 2;
            float pz = (float)z / divisions * depth - depth / 2;

            Mesh column = MeshGenerator::createCylinder(memberSize / 2, height, 8);
            column.transform(Vec3(px, height / 2, pz), Vec3(1, 1, 1), Vec3(0, 0, 0));
            mesh.merge(column);
        }
    }

    // Horizontal members
    for (int y = 0; y <= divisions; y++) {
        float py = (float)y / divisions * height;

        for (int x = 0; x <= divisions; x++) {
            float px = (float)x / divisions * width - width / 2;
            Mesh beam = MeshGenerator::createCylinder(memberSize / 2, depth, 8);
            beam.transform(Vec3(px, py, 0), Vec3(1, 1, 1), Vec3(M_PI / 2, 0, 0));
            mesh.merge(beam);
        }

        for (int z = 0; z <= divisions; z++) {
            float pz = (float)z / divisions * depth - depth / 2;
            Mesh beam = MeshGenerator::createCylinder(memberSize / 2, width, 8);
            beam.transform(Vec3(0, py, pz), Vec3(1, 1, 1), Vec3(0, 0, M_PI / 2));
            mesh.merge(beam);
        }
    }

    return mesh;
}

Mesh AlgorithmLibrary::generateGearSystem(const std::vector<AlgorithmParameter>& params) {
    int teethCount = (int)params[0].value;
    float outerRadius = params[1].value;
    float innerRadius = params[2].value;
    float thickness = params[3].value;
    float toothDepth = params[4].value;

    Mesh mesh;
    int verticesPerTooth = 6;

    // Add all vertices (bottom and top for each profile point)
    for (int i = 0; i < teethCount; i++) {
        float angle1 = 2.0f * M_PI * i / teethCount;
        float angle2 = 2.0f * M_PI * (i + 0.4f) / teethCount;
        float angle3 = 2.0f * M_PI * (i + 0.6f) / teethCount;
        float angle4 = 2.0f * M_PI * (i + 1.0f) / teethCount;

        std::vector<Vec3> profile = {
            Vec3(innerRadius * std::cos(angle1), 0, innerRadius * std::sin(angle1)),
            Vec3(innerRadius * std::cos(angle2), 0, innerRadius * std::sin(angle2)),
            Vec3((outerRadius + toothDepth) * std::cos(angle2 + 0.05f), 0, (outerRadius + toothDepth) * std::sin(angle2 + 0.05f)),
            Vec3((outerRadius + toothDepth) * std::cos(angle3 - 0.05f), 0, (outerRadius + toothDepth) * std::sin(angle3 - 0.05f)),
            Vec3(innerRadius * std::cos(angle3), 0, innerRadius * std::sin(angle3)),
            Vec3(innerRadius * std::cos(angle4), 0, innerRadius * std::sin(angle4))
        };

        for (const auto& p : profile) {
            mesh.addVertex(Vertex(Vec3(p.x, 0, p.z)));
            mesh.addVertex(Vertex(Vec3(p.x, thickness, p.z)));
        }
    }

    // Create side faces
    for (int i = 0; i < teethCount; i++) {
        int baseIdx = i * verticesPerTooth * 2;

        for (int j = 0; j < verticesPerTooth; j++) {
            int nextJ = (j + 1) % verticesPerTooth;
            int b0 = baseIdx + j * 2;
            int b1 = baseIdx + nextJ * 2;
            int t0 = b0 + 1;
            int t1 = b1 + 1;

            // Handle wrap-around for last segment
            if (j == verticesPerTooth - 1) {
                if (i < teethCount - 1) {
                    b1 = (i + 1) * verticesPerTooth * 2;
                    t1 = b1 + 1;
                } else {
                    b1 = 0;
                    t1 = 1;
                }
            }

            mesh.addTriangle(b0, b1, t0);
            mesh.addTriangle(t0, b1, t1);
        }
    }

    // Add center points for caps
    int centerBottomIdx = mesh.getVertexCount();
    mesh.addVertex(Vertex(Vec3(0, 0, 0)));
    int centerTopIdx = mesh.getVertexCount();
    mesh.addVertex(Vertex(Vec3(0, thickness, 0)));

    // Create top and bottom caps
    for (int i = 0; i < teethCount; i++) {
        int baseIdx = i * verticesPerTooth * 2;
        int nextBaseIdx = ((i + 1) % teethCount) * verticesPerTooth * 2;

        for (int j = 0; j < verticesPerTooth; j++) {
            int b0 = baseIdx + j * 2;
            int t0 = b0 + 1;
            int b1 = (j == verticesPerTooth - 1) ? nextBaseIdx : (baseIdx + (j + 1) * 2);
            int t1 = b1 + 1;

            mesh.addTriangle(centerBottomIdx, b1, b0);
            mesh.addTriangle(centerTopIdx, t0, t1);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateSpringCoil(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float coilRadius = params[1].value;
    float turns = params[2].value;
    float pitch = params[3].value;
    int segments = (int)params[4].value;

    Mesh mesh;

    int totalPoints = (int)(turns * segments);
    for (int i = 0; i <= totalPoints; i++) {
        float t = (float)i / totalPoints * turns * 2 * M_PI;
        float x = radius * std::cos(t);
        float y = pitch * t / (2 * M_PI);
        float z = radius * std::sin(t);

        for (int j = 0; j < 8; j++) {
            float angle = 2.0f * M_PI * j / 8;
            float dx = coilRadius * std::cos(angle) * std::cos(t);
            float dy = coilRadius * std::sin(angle);
            float dz = coilRadius * std::cos(angle) * std::sin(t);

            mesh.addVertex(Vertex(Vec3(x + dx, y + dy, z + dz)));
        }
    }

    for (int i = 0; i < totalPoints; i++) {
        for (int j = 0; j < 8; j++) {
            int next = (j + 1) % 8;
            int idx = i * 8 + j;
            int nextRow = idx + 8;

            mesh.addTriangle(idx, nextRow, idx + 1);
            mesh.addTriangle(idx + 1, nextRow, nextRow + 1);
        }
    }

    mesh.computeNormals();
    return mesh;
}

// Placeholder implementations for remaining algorithms
Mesh AlgorithmLibrary::generateThreadedBolt(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float length = params[1].value;
    float threadPitch = params[2].value;
    float threadDepth = params[3].value;
    int segments = (int)params[4].value;

    Mesh mesh;

    // Create bolt shaft with threads
    int heightSegments = (int)(length / threadPitch) * 4;
    int circumSegments = segments;

    for (int i = 0; i <= heightSegments; i++) {
        float h = (float)i / heightSegments * length;
        float threadAngle = h / threadPitch * 2.0f * M_PI;

        for (int j = 0; j <= circumSegments; j++) {
            float angle = 2.0f * M_PI * j / circumSegments;

            // Calculate thread profile
            float threadOffset = threadDepth * 0.5f * (1.0f + std::sin(threadAngle));
            float r = radius - threadOffset;

            float x = r * std::cos(angle);
            float z = r * std::sin(angle);
            float y = h;

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Generate triangles
    for (int i = 0; i < heightSegments; i++) {
        for (int j = 0; j < circumSegments; j++) {
            int idx0 = i * (circumSegments + 1) + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * (circumSegments + 1) + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    // Add bolt head
    Mesh head = MeshGenerator::createCylinder(radius * 1.5f, radius * 0.5f, 6);
    head.transform(Vec3(0, length, 0), Vec3(1, 1, 1), Vec3(0, 0, 0));
    mesh.merge(head);

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateBearingAssembly(const std::vector<AlgorithmParameter>& params) {
    float innerRadius = params[0].value;
    float outerRadius = params[1].value;
    int ballCount = (int)params[2].value;
    float ballRadius = params[3].value;
    float thickness = params[4].value;

    Mesh mesh;

    // Create inner ring
    Mesh innerRing = MeshGenerator::createTorus(innerRadius, thickness * 0.3f, 32, 16);
    mesh.merge(innerRing);

    // Create outer ring
    Mesh outerRing = MeshGenerator::createTorus(outerRadius, thickness * 0.3f, 32, 16);
    mesh.merge(outerRing);

    // Create ball bearings between rings
    float ballOrbitRadius = (innerRadius + outerRadius) / 2.0f;

    for (int i = 0; i < ballCount; i++) {
        float angle = 2.0f * M_PI * i / ballCount;
        float x = ballOrbitRadius * std::cos(angle);
        float z = ballOrbitRadius * std::sin(angle);

        Mesh ball = MeshGenerator::createSphere(ballRadius, 16, 8);
        ball.transform(Vec3(x, 0, z), Vec3(1, 1, 1), Vec3(0, 0, 0));
        mesh.merge(ball);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateBranchingTree(const std::vector<AlgorithmParameter>& params) {
    float trunkRadius = params[0].value;
    float length = params[1].value;
    int depth = (int)params[2].value;
    float branchAngle = params[3].value * M_PI / 180.0f;
    int branchCount = (int)params[4].value;

    Mesh mesh;

    // Recursive function to build tree
    std::function<void(Vec3, Vec3, float, float, int)> buildBranch = [&](Vec3 start, Vec3 direction, float radius, float branchLength, int currentDepth) {
        if (currentDepth <= 0 || radius < 0.01f) return;

        Vec3 end = start + direction * branchLength;

        // Create cylinder for branch
        Mesh branch = MeshGenerator::createCylinder(radius, branchLength, 6);

        // Orient branch from start to end
        Vec3 up = direction;
        up.normalize();
        float angle = std::atan2(up.x, up.y);
        float tilt = std::acos(up.z);

        branch.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(tilt, 0, angle));
        branch.transform(start, Vec3(1, 1, 1), Vec3(0, 0, 0));

        mesh.merge(branch);

        // Create child branches
        if (currentDepth > 1) {
            for (int i = 0; i < branchCount; i++) {
                float rotation = 2.0f * M_PI * i / branchCount;

                // Calculate branch direction
                Vec3 perpendicular = Vec3(std::cos(rotation), std::sin(rotation), 0);
                Vec3 newDir = direction * std::cos(branchAngle) + perpendicular * std::sin(branchAngle);
                newDir.normalize();

                buildBranch(end, newDir, radius * 0.7f, branchLength * 0.7f, currentDepth - 1);
            }
        }
    };

    buildBranch(Vec3(0, 0, 0), Vec3(0, 0, 1), trunkRadius, length, depth);

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateCoralGrowth(const std::vector<AlgorithmParameter>& params) {
    float baseRadius = params[0].value;
    float growthFactor = params[1].value;
    int iterations = (int)params[2].value;
    float randomness = params[3].value;

    Mesh mesh;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> angleDist(0.0f, 2.0f * M_PI);
    std::uniform_real_distribution<float> tiltDist(0.3f, 0.8f);
    std::uniform_real_distribution<float> randomDist(-randomness, randomness);

    // Recursive function to grow coral branches
    std::function<void(Vec3, Vec3, float, int)> growBranch = [&](Vec3 start, Vec3 direction, float radius, int depth) {
        if (depth <= 0 || radius < 0.05f) return;

        float branchLength = radius * growthFactor;
        Vec3 end = start + direction * branchLength;

        // Create branch segment as a cone
        Mesh segment = MeshGenerator::createCone(radius, branchLength, 6);

        // Orient the segment
        Vec3 up = direction;
        up.normalize();
        float angle = std::atan2(up.x, up.y);
        float tilt = std::acos(up.z);

        segment.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(tilt, 0, angle));
        segment.transform(start, Vec3(1, 1, 1), Vec3(0, 0, 0));

        mesh.merge(segment);

        // Create child branches
        if (depth > 1) {
            int branchCount = 2 + (depth % 3);

            for (int i = 0; i < branchCount; i++) {
                // Random branching angle
                float branchAngle = angleDist(gen);
                float tiltAmount = tiltDist(gen);

                // Create perpendicular direction
                Vec3 perpendicular = Vec3(std::cos(branchAngle), std::sin(branchAngle), 0);
                Vec3 newDir = direction * (1.0f - tiltAmount) + perpendicular * tiltAmount;

                // Add randomness
                newDir.x += randomDist(gen);
                newDir.y += randomDist(gen);
                newDir.z += randomDist(gen);
                newDir.normalize();

                growBranch(end, newDir, radius * 0.7f, depth - 1);
            }
        }
    };

    // Start growing from base
    growBranch(Vec3(0, 0, 0), Vec3(0, 0, 1), baseRadius, iterations);

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateShellGrowth(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    return generateShellStructure(params);
}

Mesh AlgorithmLibrary::generateFlowerPetals(const std::vector<AlgorithmParameter>& params) {
    int petalCount = (int)params[0].value;
    float petalLength = params[1].value;
    float petalWidth = params[2].value;
    float curvature = params[3].value;

    Mesh mesh;

    // Create center of flower
    Mesh center = MeshGenerator::createSphere(0.2f, 8, 6);
    mesh.merge(center);

    // Create each petal
    for (int i = 0; i < petalCount; i++) {
        float angle = 2.0f * M_PI * i / petalCount;

        Mesh petal;
        int segments = 10;
        int widthSegs = 6;

        // Generate petal shape
        for (int j = 0; j <= segments; j++) {
            float t = (float)j / segments;
            float dist = t * petalLength;

            // Petal width variation (wider in middle, narrow at tip)
            float widthFactor = std::sin(t * M_PI) * petalWidth;

            // Vertical curvature
            float height = std::sin(t * M_PI) * curvature * 0.3f;

            for (int k = 0; k <= widthSegs; k++) {
                float u = (float)k / widthSegs - 0.5f;
                float x = u * widthFactor;
                float y = dist;
                float z = height * (1.0f - std::abs(u) * 2.0f); // Curved edges

                petal.addVertex(Vertex(Vec3(x, y, z)));
            }
        }

        // Generate triangles for petal
        for (int j = 0; j < segments; j++) {
            for (int k = 0; k < widthSegs; k++) {
                int idx0 = j * (widthSegs + 1) + k;
                int idx1 = idx0 + 1;
                int idx2 = (j + 1) * (widthSegs + 1) + k;
                int idx3 = idx2 + 1;

                petal.addTriangle(idx0, idx2, idx1);
                petal.addTriangle(idx1, idx2, idx3);
            }
        }

        petal.computeNormals();

        // Rotate petal around Z axis
        petal.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(0, 0, angle));
        mesh.merge(petal);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateLeafVenation(const std::vector<AlgorithmParameter>& params) {
    float width = params[0].value;
    float length = params[1].value;
    float veinDensity = params[2].value;
    int branchCount = (int)params[3].value;

    Mesh mesh;
    int segments = 30;

    // Generate leaf shape (elliptical)
    for (int i = 0; i <= segments; i++) {
        for (int j = 0; j <= segments; j++) {
            float u = (float)i / segments - 0.5f;
            float v = (float)j / segments - 0.5f;

            // Elliptical shape with pointed tip
            float r = std::sqrt(u * u + v * v);
            float edgeFactor = std::max(0.0f, 1.0f - r * 2.0f);

            float x = u * width * 2.0f;
            float y = v * length * 2.0f * (1.0f - u * 0.3f); // Pointed shape
            float z = edgeFactor * 0.05f; // Slight curvature

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Generate triangles for leaf surface
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < segments; j++) {
            int idx0 = i * (segments + 1) + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * (segments + 1) + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    // Add vein structure
    float veinRadius = 0.02f * veinDensity;

    // Central vein (midrib)
    int veinSegments = 20;
    for (int i = 0; i < veinSegments; i++) {
        float t = (float)i / veinSegments;
        Vec3 start(0, (t - 0.5f) * length * 2.0f, 0.06f);
        Vec3 end(0, ((t + 1.0f / veinSegments) - 0.5f) * length * 2.0f, 0.06f);

        Mesh veinSeg = MeshGenerator::createCylinder(veinRadius, (end - start).length(), 4);
        veinSeg.transform(start, Vec3(1, 1, 1), Vec3(M_PI / 2, 0, 0));
        mesh.merge(veinSeg);
    }

    // Secondary veins branching from midrib
    for (int b = 0; b < branchCount; b++) {
        float branchT = 0.2f + 0.6f * b / (branchCount - 1);
        float branchY = (branchT - 0.5f) * length * 2.0f;
        float branchAngle = 45.0f * M_PI / 180.0f;

        // Left branch
        for (int side = -1; side <= 1; side += 2) {
            int subSegments = 10;
            for (int i = 0; i < subSegments; i++) {
                float t = (float)i / subSegments;
                float veinLength = width * (1.0f - branchT * 0.5f);

                Vec3 start(side * t * veinLength, branchY + t * veinLength * 0.3f, 0.06f);
                Vec3 end(side * (t + 1.0f / subSegments) * veinLength,
                         branchY + (t + 1.0f / subSegments) * veinLength * 0.3f, 0.06f);

                float segRadius = veinRadius * (1.0f - t * 0.5f);

                Mesh veinSeg = MeshGenerator::createCylinder(segRadius, (end - start).length(), 4);

                Vec3 dir = end - start;
                dir.normalize();
                float angle = std::atan2(dir.y, dir.x);

                veinSeg.transform(start, Vec3(1, 1, 1), Vec3(0, 0, angle));
                mesh.merge(veinSeg);
            }
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateMobiusStrip(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float width = params[1].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createMobiusStrip(radius, width, segments);
}

Mesh AlgorithmLibrary::generateKleinBottle(const std::vector<AlgorithmParameter>& params) {
    int segments = (int)params[1].value;
    return MeshGenerator::createKleinBottle(segments);
}

Mesh AlgorithmLibrary::generateTrefoilKnot(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createTrefoilKnot(scale, segments);
}

Mesh AlgorithmLibrary::generateLorenzAttractor(const std::vector<AlgorithmParameter>& params) {
    float sigma = params[0].value;
    float rho = params[1].value;
    float beta = params[2].value;
    int iterations = (int)params[3].value;
    float scale = params[4].value;

    Mesh mesh;
    Vec3 p(1.0f, 1.0f, 1.0f);
    float dt = 0.01f;
    float tubeRadius = 0.02f;

    std::vector<Vec3> path;
    for (int i = 0; i < iterations; i++) {
        float dx = sigma * (p.y - p.x);
        float dy = p.x * (rho - p.z) - p.y;
        float dz = p.x * p.y - beta * p.z;

        p.x += dx * dt;
        p.y += dy * dt;
        p.z += dz * dt;

        path.push_back(p * scale);
    }

    // Create a tube along the path
    int circleSegments = 6;
    for (size_t i = 0; i < path.size() - 1; i++) {
        Vec3 dir = path[i + 1] - path[i];
        if (dir.length() < 0.001f) continue;
        dir.normalize();

        // Create perpendicular vectors
        Vec3 up = (std::abs(dir.y) < 0.99f) ? Vec3(0, 1, 0) : Vec3(1, 0, 0);
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);

        // Create circle vertices
        for (int j = 0; j < circleSegments; j++) {
            float angle = 2.0f * M_PI * j / circleSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(path[i] + offset));
        }
    }

    // Create triangles
    for (size_t i = 0; i < path.size() - 2; i++) {
        for (int j = 0; j < circleSegments; j++) {
            int curr = i * circleSegments + j;
            int next = i * circleSegments + (j + 1) % circleSegments;
            int currNext = (i + 1) * circleSegments + j;
            int nextNext = (i + 1) * circleSegments + (j + 1) % circleSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateHyperbolicParaboloid(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    float curvature = params[1].value;
    int resolution = (int)params[2].value;

    Mesh mesh;

    for (int y = 0; y <= resolution; y++) {
        for (int x = 0; x <= resolution; x++) {
            float px = (float)x / resolution * size - size / 2;
            float pz = (float)y / resolution * size - size / 2;
            float py = curvature * (px * px - pz * pz) / (size * size);

            mesh.addVertex(Vertex(Vec3(px, py, pz)));
        }
    }

    for (int y = 0; y < resolution; y++) {
        for (int x = 0; x < resolution; x++) {
            int i0 = y * (resolution + 1) + x;
            int i1 = i0 + 1;
            int i2 = i0 + resolution + 1;
            int i3 = i2 + 1;

            mesh.addTriangle(i0, i2, i1);
            mesh.addTriangle(i1, i2, i3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generatePenroseTiling(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float scale = params[1].value;
    float thickness = params[2].value;

    Mesh mesh;

    // Create Penrose tiling pattern using the golden ratio
    float phi = (1.0f + std::sqrt(5.0f)) / 2.0f; // Golden ratio
    int count = 5 * (1 << iterations);

    for (int i = 0; i < count; i++) {
        float angle = 2.0f * M_PI * i / count;
        float radius = scale * (1.0f + 0.5f * std::sin(phi * angle));

        // Create rhombus tiles with thickness
        Mesh tile = MeshGenerator::createCube(1.0f);
        tile.transform(
            Vec3(radius * std::cos(angle), 0, radius * std::sin(angle)),
            Vec3(0.3f * scale, thickness, 0.5f * scale),
            Vec3(0, angle, 0)
        );
        mesh.merge(tile);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateIslamicPattern(const std::vector<AlgorithmParameter>& params) {
    int symmetry = (int)params[0].value;
    float radius = params[1].value;
    int complexity = (int)params[2].value;
    float lineWidth = params[3].value;

    Mesh mesh;

    // Create Islamic star pattern using interlocking polygons
    // Generate the main star pattern
    for (int layer = 0; layer <= complexity; layer++) {
        float layerRadius = radius * (1.0f - layer * 0.15f);
        int points = symmetry * (layer + 1);

        for (int i = 0; i < points; i++) {
            float angle1 = 2.0f * M_PI * i / points;
            float angle2 = 2.0f * M_PI * (i + 1) / points;

            // Create line segments for the pattern
            Vec3 p1(layerRadius * std::cos(angle1), 0, layerRadius * std::sin(angle1));
            Vec3 p2(layerRadius * std::cos(angle2), 0, layerRadius * std::sin(angle2));

            // Create a small cylinder for each line segment
            Vec3 dir = p2 - p1;
            float length = dir.length();
            dir.normalize();

            Mesh segment = MeshGenerator::createCylinder(lineWidth, length, 8);

            // Calculate rotation to align cylinder with line direction
            Vec3 up(0, 1, 0);
            Vec3 axis = up.cross(dir);
            float angle = std::acos(up.dot(dir));

            if (axis.length() > 0.001f) {
                axis.normalize();
                segment.transform(
                    (p1 + p2) * 0.5f,
                    Vec3(1, 1, 1),
                    axis * angle
                );
            } else {
                segment.transform((p1 + p2) * 0.5f, Vec3(1, 1, 1), Vec3(0, 0, 0));
            }

            mesh.merge(segment);

            // Add radial connections for complexity
            if (layer > 0) {
                float prevRadius = radius * (1.0f - (layer - 1) * 0.15f);
                Vec3 p3(prevRadius * std::cos(angle1), 0, prevRadius * std::sin(angle1));

                Vec3 dir2 = p3 - p1;
                float length2 = dir2.length();
                dir2.normalize();

                Mesh radial = MeshGenerator::createCylinder(lineWidth * 0.7f, length2, 6);

                Vec3 axis2 = up.cross(dir2);
                float angle2_rad = std::acos(up.dot(dir2));

                if (axis2.length() > 0.001f) {
                    axis2.normalize();
                    radial.transform(
                        (p1 + p3) * 0.5f,
                        Vec3(1, 1, 1),
                        axis2 * angle2_rad
                    );
                } else {
                    radial.transform((p1 + p3) * 0.5f, Vec3(1, 1, 1), Vec3(0, 0, 0));
                }

                mesh.merge(radial);
            }
        }

        // Add decorative nodes at vertices
        for (int i = 0; i < points; i++) {
            float angle = 2.0f * M_PI * i / points;
            Vec3 pos(layerRadius * std::cos(angle), 0, layerRadius * std::sin(angle));

            Mesh node = MeshGenerator::createSphere(lineWidth * 1.5f, 8, 6);
            node.transform(pos, Vec3(1, 1, 1), Vec3(0, 0, 0));
            mesh.merge(node);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateTruchetTiles(const std::vector<AlgorithmParameter>& params) {
    int gridSize = (int)params[0].value;
    float tileSize = params[1].value;

    Mesh mesh;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_int_distribution<int> dist(0, 3);

    // Generate Truchet tiles - quarter circles in different orientations
    for (int x = 0; x < gridSize; x++) {
        for (int y = 0; y < gridSize; y++) {
            int rotation = dist(gen); // Random rotation

            // Create a tile with a quarter circle pattern
            Mesh tile = MeshGenerator::createPlane(tileSize, tileSize, 2, 2);

            float posX = (x - gridSize / 2.0f) * tileSize;
            float posY = (y - gridSize / 2.0f) * tileSize;
            float angle = rotation * M_PI / 2.0f;

            tile.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(0, 0, angle));
            tile.transform(Vec3(posX, posY, 0), Vec3(1, 1, 1), Vec3(0, 0, 0));

            mesh.merge(tile);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModularCircles(const std::vector<AlgorithmParameter>& params) {
    int baseCount = (int)params[0].value;
    float radius = params[1].value;

    Mesh mesh;
    for (int i = 0; i < baseCount; i++) {
        float angle = 2.0f * M_PI * i / baseCount;
        float x = radius * std::cos(angle);
        float z = radius * std::sin(angle);

        Mesh circle = MeshGenerator::createTorus(0.3f, 0.1f, 16, 8);
        circle.transform(Vec3(x, 0, z), Vec3(1, 1, 1), Vec3(0, 0, 0));
        mesh.merge(circle);
    }
    return mesh;
}

Mesh AlgorithmLibrary::generateGyroidSurface(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    float thickness = params[1].value;
    int resolution = (int)params[2].value;

    SDFEngine sdf;
    sdf.setSDF([scale, thickness](const Vec3& p) {
        float x = p.x / scale;
        float y = p.y / scale;
        float z = p.z / scale;
        float gyroid = std::sin(x) * std::cos(y) + std::sin(y) * std::cos(z) + std::sin(z) * std::cos(x);
        return gyroid - thickness;
    });
    sdf.setBounds(Vec3(-scale * M_PI, -scale * M_PI, -scale * M_PI),
                  Vec3(scale * M_PI, scale * M_PI, scale * M_PI));

    return MeshGenerator::surfels(sdf, resolution);
}

Mesh AlgorithmLibrary::generateSchwarzPSurface(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    float thickness = params[1].value;
    int resolution = (int)params[2].value;

    SDFEngine sdf;
    sdf.setSDF([scale, thickness](const Vec3& p) {
        float x = p.x / scale;
        float y = p.y / scale;
        float z = p.z / scale;
        float schwarz = std::cos(x) + std::cos(y) + std::cos(z);
        return schwarz - thickness;
    });
    sdf.setBounds(Vec3(-scale * M_PI, -scale * M_PI, -scale * M_PI),
                  Vec3(scale * M_PI, scale * M_PI, scale * M_PI));

    return MeshGenerator::surfels(sdf, resolution);
}

Mesh AlgorithmLibrary::generateReactionDiffusion(const std::vector<AlgorithmParameter>& params) {
    int gridSize = (int)params[0].value;

    // Create a height-based surface using simple reaction-diffusion simulation
    std::vector<std::vector<float>> grid(gridSize, std::vector<float>(gridSize, 0.0f));
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dist(0.0f, 1.0f);

    // Initialize with random values
    for (int i = 0; i < gridSize; i++) {
        for (int j = 0; j < gridSize; j++) {
            grid[i][j] = dist(gen);
        }
    }

    // Simple diffusion iterations
    for (int iter = 0; iter < 10; iter++) {
        std::vector<std::vector<float>> newGrid = grid;
        for (int i = 1; i < gridSize - 1; i++) {
            for (int j = 1; j < gridSize - 1; j++) {
                float avg = (grid[i-1][j] + grid[i+1][j] + grid[i][j-1] + grid[i][j+1]) / 4.0f;
                newGrid[i][j] = avg * 0.5f + grid[i][j] * 0.5f;
            }
        }
        grid = newGrid;
    }

    // Create mesh from grid
    Mesh mesh;
    float cellSize = 1.0f;

    for (int i = 0; i < gridSize; i++) {
        for (int j = 0; j < gridSize; j++) {
            float x = (i - gridSize / 2.0f) * cellSize;
            float z = (j - gridSize / 2.0f) * cellSize;
            float y = grid[i][j] * 2.0f;
            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    for (int i = 0; i < gridSize - 1; i++) {
        for (int j = 0; j < gridSize - 1; j++) {
            int idx0 = i * gridSize + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * gridSize + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateCellularAutomata(const std::vector<AlgorithmParameter>& params) {
    int gridSize = (int)params[0].value;
    int generations = (int)params[1].value;
    int birthRule = (int)params[2].value;
    int survivalRule = (int)params[3].value;

    if (gridSize < 4) gridSize = 4;
    if (gridSize > 32) gridSize = 32;

    // Initialize grid
    std::vector<std::vector<std::vector<bool>>> grid(
        gridSize, std::vector<std::vector<bool>>(gridSize, std::vector<bool>(gridSize, false)));

    std::mt19937 gen(42);
    std::uniform_real_distribution<float> dis(0.0f, 1.0f);

    // Seed with center sphere
    int center = gridSize / 2;
    int radius = gridSize / 4;
    for (int x = 0; x < gridSize; x++) {
        for (int y = 0; y < gridSize; y++) {
            for (int z = 0; z < gridSize; z++) {
                int dx = x - center, dy = y - center, dz = z - center;
                if (dx*dx + dy*dy + dz*dz < radius*radius) {
                    grid[x][y][z] = dis(gen) > 0.5f;
                }
            }
        }
    }

    // Run cellular automata
    for (int gen = 0; gen < generations; gen++) {
        auto newGrid = grid;

        for (int x = 1; x < gridSize - 1; x++) {
            for (int y = 1; y < gridSize - 1; y++) {
                for (int z = 1; z < gridSize - 1; z++) {
                    // Count neighbors
                    int neighbors = 0;
                    for (int dx = -1; dx <= 1; dx++) {
                        for (int dy = -1; dy <= 1; dy++) {
                            for (int dz = -1; dz <= 1; dz++) {
                                if (dx == 0 && dy == 0 && dz == 0) continue;
                                if (grid[x + dx][y + dy][z + dz]) neighbors++;
                            }
                        }
                    }

                    // Apply rules
                    if (grid[x][y][z]) {
                        newGrid[x][y][z] = (neighbors >= survivalRule && neighbors <= survivalRule + 3);
                    } else {
                        newGrid[x][y][z] = (neighbors == birthRule);
                    }
                }
            }
        }

        grid = newGrid;
    }

    // Generate mesh
    Mesh mesh;
    float cellSize = 0.9f;
    for (int x = 0; x < gridSize; x++) {
        for (int y = 0; y < gridSize; y++) {
            for (int z = 0; z < gridSize; z++) {
                if (grid[x][y][z]) {
                    Mesh cube = MeshGenerator::createCube(cellSize);
                    Vec3 pos((float)x - gridSize/2, (float)y - gridSize/2, (float)z - gridSize/2);
                    cube.transform(pos, Vec3(1, 1, 1), Vec3(0, 0, 0));
                    mesh.merge(cube);
                }
            }
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateLSystemPlant(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float angle = params[1].value * M_PI / 180.0f;
    float lengthScale = params[2].value;
    float thickness = params[3].value;

    Mesh mesh = MeshGenerator::createCylinder(thickness, 1.0f, 8);

    for (int i = 1; i < iterations; i++) {
        Mesh branch = MeshGenerator::createCylinder(thickness * std::pow(lengthScale, i),
                                                    std::pow(lengthScale, i), 8);
        branch.transform(Vec3(0, i, 0), Vec3(1, 1, 1), Vec3(angle * i, 0, 0));
        mesh.merge(branch);
    }

    return mesh;
}

// ========== MODIFIERS ==========

Mesh AlgorithmLibrary::generateModScale(const std::vector<AlgorithmParameter>& params) {
    float scaleX = params[0].value;
    float scaleY = params[1].value;
    float scaleZ = params[2].value;

    Mesh mesh = MeshGenerator::createCube(1.0f);
    mesh.transform(Vec3(0, 0, 0), Vec3(scaleX, scaleY, scaleZ), Vec3(0, 0, 0));
    return mesh;
}

Mesh AlgorithmLibrary::generateModTranslate(const std::vector<AlgorithmParameter>& params) {
    float x = params[0].value;
    float y = params[1].value;
    float z = params[2].value;

    Mesh mesh = MeshGenerator::createSphere(0.5f, 16, 16);
    mesh.transform(Vec3(x, y, z), Vec3(1, 1, 1), Vec3(0, 0, 0));
    return mesh;
}

Mesh AlgorithmLibrary::generateModRotate(const std::vector<AlgorithmParameter>& params) {
    float rotX = params[0].value * M_PI / 180.0f;
    float rotY = params[1].value * M_PI / 180.0f;
    float rotZ = params[2].value * M_PI / 180.0f;

    Mesh mesh = MeshGenerator::createCube(1.0f);
    mesh.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(rotX, rotY, rotZ));
    return mesh;
}

Mesh AlgorithmLibrary::generateModNoise(const std::vector<AlgorithmParameter>& params) {
    float amplitude = params[0].value;
    int seed = (int)params[1].value;

    Mesh mesh = MeshGenerator::createSphere(1.0f, 32, 16);
    mesh.computeNormals();

    std::mt19937 gen(seed);
    std::uniform_real_distribution<float> dist(-1.0f, 1.0f);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float noise = dist(gen) * amplitude;
        v.position = v.position + v.normal * noise;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModPerlinNoise(const std::vector<AlgorithmParameter>& params) {
    float amplitude = params[0].value;
    float frequency = params[1].value;
    int resolution = (int)params[2].value;

    Mesh mesh = MeshGenerator::createPlane(4.0f, 4.0f, resolution, resolution);

    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dist(-1.0f, 1.0f);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float noise = dist(gen) * amplitude * std::sin(v.position.x * frequency) * std::cos(v.position.z * frequency);
        v.position.y += noise;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModPulse(const std::vector<AlgorithmParameter>& params) {
    float amplitude = params[0].value;
    float frequency = params[1].value;

    Mesh mesh = MeshGenerator::createSphere(1.0f, 32, 16);
    mesh.computeNormals();

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float distance = v.position.length();
        float pulse = amplitude * std::sin(distance * frequency * M_PI);
        v.position = v.position + v.normal * pulse;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModRepeatDomain(const std::vector<AlgorithmParameter>& params) {
    int repeatX = (int)params[0].value;
    int repeatY = (int)params[1].value;
    int repeatZ = (int)params[2].value;
    float spacing = params[3].value;

    Mesh baseMesh = MeshGenerator::createSphere(0.3f, 16, 8);
    Mesh mesh;

    for (int x = 0; x < repeatX; x++) {
        for (int y = 0; y < repeatY; y++) {
            for (int z = 0; z < repeatZ; z++) {
                Vec3 offset(x * spacing, y * spacing, z * spacing);

                // Copy vertices with offset
                size_t baseVertexCount = mesh.getVertexCount();
                for (size_t i = 0; i < baseMesh.getVertexCount(); i++) {
                    Vertex v = baseMesh.getVertex(i);
                    v.position = v.position + offset;
                    mesh.addVertex(v);
                }

                // Copy triangles with adjusted indices
                for (size_t i = 0; i < baseMesh.getTriangleCount(); i++) {
                    Triangle t = baseMesh.getTriangle(i);
                    mesh.addTriangle(t.v0 + baseVertexCount, t.v1 + baseVertexCount, t.v2 + baseVertexCount);
                }
            }
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModSmooth(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float factor = params[1].value;

    Mesh mesh = MeshGenerator::createSphere(1.0f, 16, 8);

    for (int iter = 0; iter < iterations; iter++) {
        for (size_t i = 0; i < mesh.getVertexCount(); i++) {
            Vertex& v = mesh.getVertex(i);
            Vec3 avgPos = v.position * (1.0f - factor);

            if (i > 0) avgPos = avgPos + mesh.getVertex(i - 1).position * factor * 0.5f;
            if (i < mesh.getVertexCount() - 1) avgPos = avgPos + mesh.getVertex(i + 1).position * factor * 0.5f;

            v.position = avgPos;
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModInflate(const std::vector<AlgorithmParameter>& params) {
    float amount = params[0].value;

    Mesh mesh = MeshGenerator::createSphere(1.0f, 16, 16);
    mesh.computeNormals();

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        v.position = v.position + v.normal * amount;
    }

    return mesh;
}

Mesh AlgorithmLibrary::generateModSolidify(const std::vector<AlgorithmParameter>& params) {
    float thickness = params[0].value;

    Mesh mesh = MeshGenerator::createPlane(2.0f, 2.0f, 10, 10);
    Mesh inner = mesh;

    for (size_t i = 0; i < inner.getVertexCount(); i++) {
        Vertex& v = inner.getVertex(i);
        v.position.y -= thickness;
    }

    mesh.merge(inner);
    return mesh;
}

Mesh AlgorithmLibrary::generateModSubdivide(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;

    Mesh mesh = MeshGenerator::createCube(1.0f);

    for (int i = 0; i < iterations; i++) {
        MeshGenerator::subdivide(mesh, 1);
    }

    return mesh;
}

Mesh AlgorithmLibrary::generateModDecimate(const std::vector<AlgorithmParameter>& params) {
    float ratio = params[0].value;

    Mesh mesh = MeshGenerator::createSphere(1.0f, 32, 32);

    // Simple decimation by skipping vertices
    Mesh decimated;
    int skip = (int)(1.0f / ratio);

    for (size_t i = 0; i < mesh.getVertexCount(); i += skip) {
        decimated.addVertex(mesh.getVertex(i));
    }

    return decimated;
}

Mesh AlgorithmLibrary::generateModTwistDeform(const std::vector<AlgorithmParameter>& params) {
    float amount = params[0].value;

    Mesh mesh = MeshGenerator::createCylinder(0.5f, 2.0f, 16);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float angle = v.position.y * amount;
        float c = std::cos(angle);
        float s = std::sin(angle);

        Vec3 twisted;
        twisted.x = v.position.x * c - v.position.z * s;
        twisted.y = v.position.y;
        twisted.z = v.position.x * s + v.position.z * c;

        v.position = twisted;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModBendDeform(const std::vector<AlgorithmParameter>& params) {
    float angle = params[0].value * M_PI / 180.0f;

    Mesh mesh = MeshGenerator::createCylinder(0.5f, 3.0f, 16);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float bendAngle = v.position.y * angle / 3.0f;

        Vec3 bent;
        bent.x = v.position.x * std::cos(bendAngle) - v.position.y * std::sin(bendAngle);
        bent.y = v.position.x * std::sin(bendAngle) + v.position.y * std::cos(bendAngle);
        bent.z = v.position.z;

        v.position = bent;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModTaperDeform(const std::vector<AlgorithmParameter>& params) {
    float amount = params[0].value;

    Mesh mesh = MeshGenerator::createCylinder(0.5f, 2.0f, 16);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float scale = 1.0f - (v.position.y + 1.0f) * amount * 0.5f;
        v.position.x *= scale;
        v.position.z *= scale;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModWaveDeform(const std::vector<AlgorithmParameter>& params) {
    float amplitude = params[0].value;
    float frequency = params[1].value;

    Mesh mesh = MeshGenerator::createPlane(4.0f, 4.0f, 32, 32);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        v.position.y += amplitude * std::sin(v.position.x * frequency) * std::cos(v.position.z * frequency);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModSpherify(const std::vector<AlgorithmParameter>& params) {
    float amount = params[0].value;

    Mesh mesh = MeshGenerator::createCube(1.0f);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        Vec3 normalized = v.position;
        normalized.normalize();

        v.position = v.position * (1.0f - amount) + normalized * amount;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModLatticeDeform(const std::vector<AlgorithmParameter>& params) {
    float strength = params[0].value;
    int resolution = (int)params[1].value;

    Mesh mesh = MeshGenerator::createSphere(1.0f, 16, 16);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float latticeX = std::sin(v.position.x * resolution) * strength;
        float latticeY = std::sin(v.position.y * resolution) * strength;
        float latticeZ = std::sin(v.position.z * resolution) * strength;

        v.position = v.position + Vec3(latticeX, latticeY, latticeZ);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModMirror(const std::vector<AlgorithmParameter>& params) {
    int axis = (int)params[0].value; // 0=X, 1=Y, 2=Z

    Mesh mesh = MeshGenerator::createCube(1.0f);
    Mesh mirrored = mesh;

    for (size_t i = 0; i < mirrored.getVertexCount(); i++) {
        Vertex& v = mirrored.getVertex(i);
        if (axis == 0) v.position.x = -v.position.x;
        else if (axis == 1) v.position.y = -v.position.y;
        else v.position.z = -v.position.z;
    }

    mesh.merge(mirrored);
    return mesh;
}

Mesh AlgorithmLibrary::generateModArray(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    float spacing = params[1].value;

    Mesh result;
    Mesh base = MeshGenerator::createSphere(0.3f, 8, 8);

    for (int i = 0; i < count; i++) {
        Mesh copy = base;
        copy.transform(Vec3(i * spacing, 0, 0), Vec3(1, 1, 1), Vec3(0, 0, 0));
        result.merge(copy);
    }

    return result;
}

Mesh AlgorithmLibrary::generateModShell(const std::vector<AlgorithmParameter>& params) {
    float thickness = params[0].value;

    Mesh outer = MeshGenerator::createSphere(1.0f, 16, 16);
    Mesh inner = MeshGenerator::createSphere(1.0f - thickness, 16, 16);

    outer.merge(inner);
    return outer;
}

Mesh AlgorithmLibrary::generateModDisplace(const std::vector<AlgorithmParameter>& params) {
    float strength = params[0].value;
    float scale = params[1].value;

    Mesh mesh = MeshGenerator::createPlane(4.0f, 4.0f, 32, 32);

    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dist(0.0f, 1.0f);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float pattern = std::sin(v.position.x * scale) * std::cos(v.position.z * scale);
        v.position.y += pattern * strength + dist(gen) * strength * 0.2f;
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateModRemesh(const std::vector<AlgorithmParameter>& params) {
    int resolution = (int)params[0].value;

    // Create a clean remeshed sphere
    return MeshGenerator::createSphere(1.0f, resolution, resolution / 2);
}

Mesh AlgorithmLibrary::generateModExtrude(const std::vector<AlgorithmParameter>& params) {
    float distance = params[0].value;

    Mesh mesh = MeshGenerator::createPlane(2.0f, 2.0f, 1, 1);
    MeshGenerator::extrude(mesh, Vec3(0, distance, 0), distance);

    return mesh;
}

// ========== PRIMITIVES ==========

Mesh AlgorithmLibrary::generatePrimSphere(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    int segments = (int)params[1].value;
    return MeshGenerator::createSphere(radius, segments, segments / 2);
}

Mesh AlgorithmLibrary::generatePrimCube(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    return MeshGenerator::createCube(size);
}

Mesh AlgorithmLibrary::generatePrimTorus(const std::vector<AlgorithmParameter>& params) {
    float majorRadius = params[0].value;
    float minorRadius = params[1].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createTorus(majorRadius, minorRadius, segments, segments / 2);
}

Mesh AlgorithmLibrary::generatePrimCylinder(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createCylinder(radius, height, segments);
}

Mesh AlgorithmLibrary::generatePrimCone(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createCone(radius, height, segments);
}

Mesh AlgorithmLibrary::generatePrimPlane(const std::vector<AlgorithmParameter>& params) {
    float width = params[0].value;
    float height = params[1].value;
    int widthSegments = (int)params[2].value;
    int heightSegments = (int)params[3].value;
    return MeshGenerator::createPlane(width, height, widthSegments, heightSegments);
}

// ========== ADDITIONAL PRIMITIVES ==========

Mesh AlgorithmLibrary::generatePrimSuperellipsoid(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float n = params[1].value;
    float e = params[2].value;
    int segments = (int)params[3].value;

    Mesh mesh;

    // Sign function that preserves sign
    auto signedPow = [](float val, float exp) {
        return std::copysign(std::pow(std::abs(val), exp), val);
    };

    // Generate vertices
    for (int i = 0; i <= segments; i++) {
        float v = M_PI * (float)i / segments - M_PI / 2.0f;

        for (int j = 0; j <= segments; j++) {
            float u = 2.0f * M_PI * (float)j / segments;

            // Superellipsoid parametric equations
            float cosV = std::cos(v);
            float sinV = std::sin(v);
            float cosU = std::cos(u);
            float sinU = std::sin(u);

            float x = radius * signedPow(cosV, n) * signedPow(cosU, e);
            float y = radius * signedPow(cosV, n) * signedPow(sinU, e);
            float z = radius * signedPow(sinV, n);

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Generate triangles
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < segments; j++) {
            int idx0 = i * (segments + 1) + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * (segments + 1) + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generatePrimSupertoroid(const std::vector<AlgorithmParameter>& params) {
    float majorRadius = params[0].value;
    float minorRadius = params[1].value;
    float n = params[2].value;
    int segments = (int)params[3].value;

    Mesh mesh;

    // Sign function that preserves sign
    auto signedPow = [](float val, float exp) {
        return std::copysign(std::pow(std::abs(val), exp), val);
    };

    // Generate vertices
    for (int i = 0; i <= segments; i++) {
        float v = 2.0f * M_PI * (float)i / segments;

        for (int j = 0; j <= segments; j++) {
            float u = 2.0f * M_PI * (float)j / segments;

            // Supertoroid parametric equations
            float cosV = std::cos(v);
            float sinV = std::sin(v);
            float cosU = std::cos(u);
            float sinU = std::sin(u);

            float r = majorRadius + minorRadius * signedPow(cosV, n);

            float x = r * signedPow(cosU, n);
            float y = r * signedPow(sinU, n);
            float z = minorRadius * signedPow(sinV, n);

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Generate triangles
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < segments; j++) {
            int idx0 = i * (segments + 1) + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * (segments + 1) + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generatePrimUtahTeapot(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    int subdivisions = (int)params[1].value;
    return MeshGenerator::createUtahTeapot(scale, subdivisions);
}

Mesh AlgorithmLibrary::generatePrimOilTank(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createOilTank(radius, height, segments);
}

Mesh AlgorithmLibrary::generatePrimGeosphere(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    int subdivisions = (int)params[1].value;
    return MeshGenerator::createGeoSphere(radius, subdivisions);
}

Mesh AlgorithmLibrary::generatePrimTube(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    float thickness = params[2].value;
    int segments = (int)params[3].value;
    return MeshGenerator::createTube(radius, height, thickness, segments);
}

Mesh AlgorithmLibrary::generatePrimPyramid(const std::vector<AlgorithmParameter>& params) {
    float baseSize = params[0].value;
    float height = params[1].value;
    int subdivisions = (int)params[2].value;
    return MeshGenerator::createPyramid(baseSize, height, subdivisions);
}

Mesh AlgorithmLibrary::generatePrimHedra(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    int sides = (int)params[1].value;
    int subdivision = (int)params[2].value;
    return MeshGenerator::createHedra(radius, sides, subdivision);
}

Mesh AlgorithmLibrary::generatePrimTorusKnot(const std::vector<AlgorithmParameter>& params) {
    int p = (int)params[0].value;
    int q = (int)params[1].value;
    float tubeRadius = params[2].value;
    int segments = (int)params[3].value;
    return MeshGenerator::createTorusKnot(p, q, tubeRadius, segments);
}


Mesh AlgorithmLibrary::generatePrimCapsule(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createCapsule(radius, height, segments);
}

Mesh AlgorithmLibrary::generatePrimSpindle(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    int segments = (int)params[2].value;
    return MeshGenerator::createSpindle(radius, height, segments);
}

Mesh AlgorithmLibrary::generatePrimLExt(const std::vector<AlgorithmParameter>& params) {
    float width = params[0].value;
    float height = params[1].value;
    float thickness = params[2].value;
    float depth = params[3].value;
    return MeshGenerator::createLExt(width, height, thickness, depth);
}

Mesh AlgorithmLibrary::generatePrimCExt(const std::vector<AlgorithmParameter>& params) {
    float width = params[0].value;
    float height = params[1].value;
    float thickness = params[2].value;
    float depth = params[3].value;
    return MeshGenerator::createCExt(width, height, thickness, depth);
}

Mesh AlgorithmLibrary::generatePrimGengon(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    int sides = (int)params[1].value;
    float fillet = params[2].value;
    float height = params[3].value;
    return MeshGenerator::createGengon(radius, sides, fillet, height);
}

Mesh AlgorithmLibrary::generatePrimRingWave(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float thickness = params[1].value;
    float waveAmplitude = params[2].value;
    int waveFrequency = (int)params[3].value;
    int segments = (int)params[4].value;
    return MeshGenerator::createRingWave(radius, thickness, waveAmplitude, waveFrequency, segments);
}

Mesh AlgorithmLibrary::generatePrimHose(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float length = params[1].value;
    float bendAngle = params[2].value;
    int segments = (int)params[3].value;
    return MeshGenerator::createHose(radius, length, bendAngle, segments);
}

Mesh AlgorithmLibrary::generatePrimPrism(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    int sides = (int)params[1].value;
    float height = params[2].value;
    int heightSegments = (int)params[3].value;
    return MeshGenerator::createPrism(radius, sides, height, heightSegments);
}

// ========== PLATONIC SOLIDS ==========

Mesh AlgorithmLibrary::generatePlatoTetrahedron(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    return MeshGenerator::createTetrahedron(size);
}

Mesh AlgorithmLibrary::generatePlatoOctahedron(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    return MeshGenerator::createOctahedron(size);
}

Mesh AlgorithmLibrary::generatePlatoIcosahedron(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    return MeshGenerator::createIcosahedron(size);
}

Mesh AlgorithmLibrary::generatePlatoDodecahedron(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    return MeshGenerator::createDodecahedron(size);
}

// ========== PARAMETRIC CURVES ==========

Mesh AlgorithmLibrary::generateParamLissajous(const std::vector<AlgorithmParameter>& params) {
    float a = params[0].value;
    float b = params[1].value;
    float c = params[2].value;
    int segments = (int)params[3].value;

    // Generate curve points
    std::vector<Vec3> curvePoints;
    for (int i = 0; i <= segments; i++) {
        float t = 2.0f * M_PI * i / segments;
        float x = std::sin(a * t);
        float y = std::sin(b * t);
        float z = std::sin(c * t);
        curvePoints.push_back(Vec3(x, y, z));
    }

    // Create tube along curve
    Mesh mesh;
    float tubeRadius = 0.05f;
    int tubeSegments = 8;

    for (size_t i = 0; i < curvePoints.size() - 1; i++) {
        Vec3 p1 = curvePoints[i];
        Vec3 p2 = curvePoints[i + 1];
        Vec3 dir = p2 - p1;

        // Create perpendicular vectors
        Vec3 up = (std::abs(dir.y) < 0.9f) ? Vec3(0, 1, 0) : Vec3(1, 0, 0);
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);
        up.normalize();

        // Create ring of vertices
        for (int j = 0; j < tubeSegments; j++) {
            float angle = 2.0f * M_PI * j / tubeSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(p1 + offset));
        }
    }

    // Create triangles
    for (int i = 0; i < (int)curvePoints.size() - 2; i++) {
        for (int j = 0; j < tubeSegments; j++) {
            int curr = i * tubeSegments + j;
            int next = i * tubeSegments + (j + 1) % tubeSegments;
            int currNext = (i + 1) * tubeSegments + j;
            int nextNext = (i + 1) * tubeSegments + (j + 1) % tubeSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateParamRose(const std::vector<AlgorithmParameter>& params) {
    float n = params[0].value;
    float d = params[1].value;
    float scale = params[2].value;
    int segments = (int)params[3].value;

    // Generate curve points
    std::vector<Vec3> curvePoints;
    for (int i = 0; i <= segments; i++) {
        float theta = 2.0f * M_PI * i / segments;
        float r = std::cos(n / d * theta) * scale;
        float x = r * std::cos(theta);
        float y = r * std::sin(theta);
        curvePoints.push_back(Vec3(x, y, 0));
    }

    // Create tube along curve
    Mesh mesh;
    float tubeRadius = 0.05f;
    int tubeSegments = 8;

    for (size_t i = 0; i < curvePoints.size() - 1; i++) {
        Vec3 p1 = curvePoints[i];
        Vec3 p2 = curvePoints[i + 1];
        Vec3 dir = p2 - p1;

        if (dir.length() < 0.001f) continue; // Skip degenerate segments

        // Create perpendicular vectors
        Vec3 up = Vec3(0, 0, 1); // For 2D curve in XY plane
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);
        up.normalize();

        // Create ring of vertices
        for (int j = 0; j < tubeSegments; j++) {
            float angle = 2.0f * M_PI * j / tubeSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(p1 + offset));
        }
    }

    // Create triangles
    int numRings = 0;
    for (size_t i = 0; i < curvePoints.size() - 1; i++) {
        Vec3 dir = curvePoints[i + 1] - curvePoints[i];
        if (dir.length() >= 0.001f) numRings++;
    }

    for (int i = 0; i < numRings - 1; i++) {
        for (int j = 0; j < tubeSegments; j++) {
            int curr = i * tubeSegments + j;
            int next = i * tubeSegments + (j + 1) % tubeSegments;
            int currNext = (i + 1) * tubeSegments + j;
            int nextNext = (i + 1) * tubeSegments + (j + 1) % tubeSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateParamHelix(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    float turns = params[2].value;
    int segments = (int)params[3].value;

    // Generate curve points
    std::vector<Vec3> curvePoints;
    for (int i = 0; i <= segments; i++) {
        float t = turns * 2.0f * M_PI * i / segments;
        float x = radius * std::cos(t);
        float z = radius * std::sin(t);
        float y = height * i / segments;
        curvePoints.push_back(Vec3(x, y, z));
    }

    // Create tube along curve
    Mesh mesh;
    float tubeRadius = 0.08f;
    int tubeSegments = 8;

    for (size_t i = 0; i < curvePoints.size() - 1; i++) {
        Vec3 p1 = curvePoints[i];
        Vec3 p2 = curvePoints[i + 1];
        Vec3 dir = p2 - p1;
        dir.normalize();

        // Create perpendicular vectors
        Vec3 up = (std::abs(dir.y) < 0.9f) ? Vec3(0, 1, 0) : Vec3(1, 0, 0);
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);
        up.normalize();

        // Create ring of vertices
        for (int j = 0; j < tubeSegments; j++) {
            float angle = 2.0f * M_PI * j / tubeSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(p1 + offset));
        }
    }

    // Create triangles
    for (int i = 0; i < (int)curvePoints.size() - 2; i++) {
        for (int j = 0; j < tubeSegments; j++) {
            int curr = i * tubeSegments + j;
            int next = i * tubeSegments + (j + 1) % tubeSegments;
            int currNext = (i + 1) * tubeSegments + j;
            int nextNext = (i + 1) * tubeSegments + (j + 1) % tubeSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

// ========== TOPOLOGY ==========

Mesh AlgorithmLibrary::generateTopoFigureEightKnot(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    float tubeRadius = params[1].value;
    int segments = (int)params[2].value;

    // Generate curve points for figure-eight knot
    std::vector<Vec3> curvePoints;
    for (int i = 0; i <= segments; i++) {
        float t = 2.0f * M_PI * i / segments;
        float x = (2.0f + std::cos(2.0f * t)) * std::cos(3.0f * t) * scale;
        float y = (2.0f + std::cos(2.0f * t)) * std::sin(3.0f * t) * scale;
        float z = std::sin(4.0f * t) * scale;
        curvePoints.push_back(Vec3(x, y, z));
    }

    // Create tube along curve
    Mesh mesh;
    int tubeSegments = 12;

    for (size_t i = 0; i < curvePoints.size(); i++) {
        size_t nextIdx = (i + 1) % curvePoints.size();
        Vec3 p1 = curvePoints[i];
        Vec3 p2 = curvePoints[nextIdx];
        Vec3 dir = p2 - p1;
        dir.normalize();

        // Create perpendicular vectors
        Vec3 up = (std::abs(dir.y) < 0.9f) ? Vec3(0, 1, 0) : Vec3(1, 0, 0);
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);
        up.normalize();

        // Create ring of vertices
        for (int j = 0; j < tubeSegments; j++) {
            float angle = 2.0f * M_PI * j / tubeSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(p1 + offset));
        }
    }

    // Create triangles
    for (int i = 0; i < (int)curvePoints.size(); i++) {
        for (int j = 0; j < tubeSegments; j++) {
            int curr = i * tubeSegments + j;
            int next = i * tubeSegments + (j + 1) % tubeSegments;
            int currNext = ((i + 1) % curvePoints.size()) * tubeSegments + j;
            int nextNext = ((i + 1) % curvePoints.size()) * tubeSegments + (j + 1) % tubeSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateTopoHeart(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    int segments = (int)params[1].value;

    // Generate 2D heart curve points
    std::vector<Vec3> curvePoints;
    for (int i = 0; i <= segments; i++) {
        float t = 2.0f * M_PI * i / segments;
        float x = 16.0f * std::pow(std::sin(t), 3.0f) * scale * 0.1f;
        float y = (13.0f * std::cos(t) - 5.0f * std::cos(2.0f * t) - 2.0f * std::cos(3.0f * t) - std::cos(4.0f * t)) * scale * 0.1f;
        curvePoints.push_back(Vec3(x, y, 0));
    }

    // Create tube along curve
    Mesh mesh;
    float tubeRadius = 0.08f * scale;
    int tubeSegments = 8;

    for (size_t i = 0; i < curvePoints.size(); i++) {
        size_t nextIdx = (i + 1) % curvePoints.size();
        Vec3 p1 = curvePoints[i];
        Vec3 p2 = curvePoints[nextIdx];
        Vec3 dir = p2 - p1;

        if (dir.length() < 0.001f) continue;
        dir.normalize();

        // Create perpendicular vectors (for 2D curve in XY plane)
        Vec3 up = Vec3(0, 0, 1);
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);
        up.normalize();

        // Create ring of vertices
        for (int j = 0; j < tubeSegments; j++) {
            float angle = 2.0f * M_PI * j / tubeSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(p1 + offset));
        }
    }

    // Create triangles
    int numValidRings = 0;
    for (size_t i = 0; i < curvePoints.size(); i++) {
        size_t nextIdx = (i + 1) % curvePoints.size();
        Vec3 dir = curvePoints[nextIdx] - curvePoints[i];
        if (dir.length() >= 0.001f) numValidRings++;
    }

    for (int i = 0; i < numValidRings; i++) {
        for (int j = 0; j < tubeSegments; j++) {
            int curr = i * tubeSegments + j;
            int next = i * tubeSegments + (j + 1) % tubeSegments;
            int currNext = ((i + 1) % numValidRings) * tubeSegments + j;
            int nextNext = ((i + 1) % numValidRings) * tubeSegments + (j + 1) % tubeSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

// ========== MINIMAL SURFACES ==========

Mesh AlgorithmLibrary::generateMinNeovius(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    int resolution = (int)params[2].value;

    SDFEngine sdf;
    sdf.setSDF([scale](const Vec3& p) {
        float x = p.x / scale;
        float y = p.y / scale;
        float z = p.z / scale;
        return 3.0f * (std::cos(x) + std::cos(y) + std::cos(z)) + 4.0f * std::cos(x) * std::cos(y) * std::cos(z);
    });
    sdf.setBounds(Vec3(-scale * M_PI, -scale * M_PI, -scale * M_PI),
                  Vec3(scale * M_PI, scale * M_PI, scale * M_PI));

    return MeshGenerator::surfels(sdf, resolution);
}

Mesh AlgorithmLibrary::generateMinHelicoid(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    float turns = params[2].value;
    int segments = (int)params[3].value;

    Mesh mesh;
    int radialSegments = segments / 4;

    // Generate vertices
    for (int i = 0; i <= segments; i++) {
        for (int j = 0; j <= radialSegments; j++) {
            float u = 2.0f * M_PI * i / segments * turns;
            float v = (float)j / radialSegments * radius;

            float x = v * std::cos(u);
            float y = v * std::sin(u);
            float z = height * u / (2.0f * M_PI * turns);

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Generate triangles
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < radialSegments; j++) {
            int idx0 = i * (radialSegments + 1) + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * (radialSegments + 1) + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateMinCatenoid(const std::vector<AlgorithmParameter>& params) {
    float radius = params[0].value;
    float height = params[1].value;
    int segments = (int)params[2].value;

    Mesh mesh;

    // Generate vertices
    for (int i = 0; i <= segments; i++) {
        for (int j = 0; j <= segments; j++) {
            float u = 2.0f * M_PI * i / segments;
            float v = (float)j / segments * height - height / 2;

            float r = radius * std::cosh(v / radius);
            float x = r * std::cos(u);
            float z = r * std::sin(u);
            float y = v;

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Generate triangles
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < segments; j++) {
            int idx0 = i * (segments + 1) + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * (segments + 1) + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateMinEnneper(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    int resolution = (int)params[1].value;

    Mesh mesh;

    // Generate vertices
    for (int i = 0; i <= resolution; i++) {
        for (int j = 0; j <= resolution; j++) {
            float u = ((float)i / resolution - 0.5f) * 4.0f * scale;
            float v = ((float)j / resolution - 0.5f) * 4.0f * scale;

            float x = u - u*u*u/3.0f + u*v*v;
            float y = v - v*v*v/3.0f + v*u*u;
            float z = u*u - v*v;

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Generate triangles
    for (int i = 0; i < resolution; i++) {
        for (int j = 0; j < resolution; j++) {
            int idx0 = i * (resolution + 1) + j;
            int idx1 = idx0 + 1;
            int idx2 = (i + 1) * (resolution + 1) + j;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

// ========== ORGANIC ==========

Mesh AlgorithmLibrary::generateOrgShellConch(const std::vector<AlgorithmParameter>& params) {
    return generateShellStructure(params);
}

Mesh AlgorithmLibrary::generateOrgCrystal(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    int facets = (int)params[1].value;

    if (facets < 4) facets = 4;
    if (facets > 12) facets = 12;

    Mesh mesh;

    // Create crystal with pyramidal points at top and bottom
    // Base is a regular polygon with 'facets' sides
    std::vector<Vec3> baseVertices;
    std::vector<Vec3> topVertices;

    float baseRadius = size * 0.5f;
    float baseHeight = -size * 0.3f;
    float topHeight = size * 0.7f;

    // Generate base ring
    for (int i = 0; i < facets; i++) {
        float angle = 2.0f * M_PI * i / facets;
        float x = baseRadius * std::cos(angle);
        float z = baseRadius * std::sin(angle);
        baseVertices.push_back(Vec3(x, baseHeight, z));
    }

    // Generate middle ring (larger)
    std::vector<Vec3> midVertices;
    float midRadius = size * 0.6f;
    float midHeight = size * 0.1f;

    for (int i = 0; i < facets; i++) {
        float angle = 2.0f * M_PI * i / facets + M_PI / facets; // Offset by half
        float x = midRadius * std::cos(angle);
        float z = midRadius * std::sin(angle);
        midVertices.push_back(Vec3(x, midHeight, z));
    }

    // Top point
    Vec3 topPoint(0, topHeight, 0);
    Vec3 bottomPoint(0, baseHeight - size * 0.2f, 0);

    // Add all vertices to mesh
    int bottomIdx = mesh.getVertexCount();
    mesh.addVertex(Vertex(bottomPoint));

    std::vector<int> baseIndices;
    for (const Vec3& v : baseVertices) {
        baseIndices.push_back(mesh.getVertexCount());
        mesh.addVertex(Vertex(v));
    }

    std::vector<int> midIndices;
    for (const Vec3& v : midVertices) {
        midIndices.push_back(mesh.getVertexCount());
        mesh.addVertex(Vertex(v));
    }

    int topIdx = mesh.getVertexCount();
    mesh.addVertex(Vertex(topPoint));

    // Create bottom pyramid faces
    for (int i = 0; i < facets; i++) {
        int next = (i + 1) % facets;
        mesh.addTriangle(bottomIdx, baseIndices[next], baseIndices[i]);
    }

    // Create lower body faces (base to mid)
    for (int i = 0; i < facets; i++) {
        int next = (i + 1) % facets;
        mesh.addTriangle(baseIndices[i], baseIndices[next], midIndices[i]);
        mesh.addTriangle(midIndices[i], baseIndices[next], midIndices[next]);
    }

    // Create upper body faces (mid to top)
    for (int i = 0; i < facets; i++) {
        int next = (i + 1) % facets;
        mesh.addTriangle(midIndices[i], topIdx, midIndices[next]);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateOrgTentacle(const std::vector<AlgorithmParameter>& params) {
    float length = params[0].value;
    float radius = params[1].value;
    int segments = (int)params[2].value;
    float twist = params[3].value;

    Mesh mesh;
    int circumSegments = 8;

    // Generate vertices
    for (int i = 0; i <= segments; i++) {
        float t = (float)i / segments;
        float r = radius * (1.0f - t * 0.5f); // Taper towards the tip
        float twistAngle = twist * t * 2.0f * M_PI;

        for (int j = 0; j < circumSegments; j++) {
            float angle = 2.0f * M_PI * j / circumSegments + twistAngle;
            float x = r * std::cos(angle);
            float z = r * std::sin(angle);
            float y = length * t;

            // Add some curvature
            float curveX = std::sin(t * M_PI) * 0.3f * length;

            mesh.addVertex(Vertex(Vec3(x + curveX, y, z)));
        }
    }

    // Generate triangles
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < circumSegments; j++) {
            int idx0 = i * circumSegments + j;
            int idx1 = i * circumSegments + (j + 1) % circumSegments;
            int idx2 = (i + 1) * circumSegments + j;
            int idx3 = (i + 1) * circumSegments + (j + 1) % circumSegments;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

// ========== FRACTALS ==========

Mesh AlgorithmLibrary::generateFracMountain(const std::vector<AlgorithmParameter>& params) {
    int size = (int)params[0].value;
    float height = params[1].value;
    float roughness = params[2].value;
    int iterations = (int)params[3].value;

    // Create a grid using power of 2 + 1
    int gridSize = (1 << iterations) + 1;

    // Initialize height map
    std::vector<std::vector<float>> heightMap(gridSize, std::vector<float>(gridSize, 0.0f));

    // Set corner values
    heightMap[0][0] = 0;
    heightMap[0][gridSize - 1] = 0;
    heightMap[gridSize - 1][0] = 0;
    heightMap[gridSize - 1][gridSize - 1] = 0;

    // Diamond-square algorithm
    std::random_device rd;
    std::mt19937 gen(rd());

    int step = gridSize - 1;
    float scale = height;

    while (step > 1) {
        int halfStep = step / 2;

        // Diamond step
        for (int y = halfStep; y < gridSize; y += step) {
            for (int x = halfStep; x < gridSize; x += step) {
                float avg = (heightMap[y - halfStep][x - halfStep] +
                            heightMap[y - halfStep][x + halfStep] +
                            heightMap[y + halfStep][x - halfStep] +
                            heightMap[y + halfStep][x + halfStep]) / 4.0f;

                std::uniform_real_distribution<float> dist(-scale, scale);
                heightMap[y][x] = avg + dist(gen);
            }
        }

        // Square step
        for (int y = 0; y < gridSize; y += halfStep) {
            for (int x = (y + halfStep) % step; x < gridSize; x += step) {
                float sum = 0;
                int count = 0;

                if (y >= halfStep) { sum += heightMap[y - halfStep][x]; count++; }
                if (y + halfStep < gridSize) { sum += heightMap[y + halfStep][x]; count++; }
                if (x >= halfStep) { sum += heightMap[y][x - halfStep]; count++; }
                if (x + halfStep < gridSize) { sum += heightMap[y][x + halfStep]; count++; }

                std::uniform_real_distribution<float> dist(-scale, scale);
                heightMap[y][x] = sum / count + dist(gen);
            }
        }

        step = halfStep;
        scale *= roughness;
    }

    // Create mesh from height map
    Mesh mesh;

    float cellSize = (float)size / (gridSize - 1);

    for (int y = 0; y < gridSize; y++) {
        for (int x = 0; x < gridSize; x++) {
            float posX = (x - gridSize / 2.0f) * cellSize;
            float posZ = (y - gridSize / 2.0f) * cellSize;
            float posY = heightMap[y][x];
            mesh.addVertex(Vertex(Vec3(posX, posY, posZ)));
        }
    }

    // Create triangles
    for (int y = 0; y < gridSize - 1; y++) {
        for (int x = 0; x < gridSize - 1; x++) {
            int idx0 = y * gridSize + x;
            int idx1 = idx0 + 1;
            int idx2 = (y + 1) * gridSize + x;
            int idx3 = idx2 + 1;

            mesh.addTriangle(idx0, idx2, idx1);
            mesh.addTriangle(idx1, idx2, idx3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateFracFern(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    int iterations = (int)params[1].value;
    float thickness = params[2].value;

    // Generate fern points using Barnsley fern IFS
    std::vector<Vec3> points;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dist(0.0f, 1.0f);

    float x = 0, y = 0;

    for (int i = 0; i < iterations; i++) {
        float r = dist(gen);
        float newX, newY;

        if (r < 0.01f) {
            // Stem
            newX = 0;
            newY = 0.16f * y;
        } else if (r < 0.86f) {
            // Smaller leaflets
            newX = 0.85f * x + 0.04f * y;
            newY = -0.04f * x + 0.85f * y + 1.6f;
        } else if (r < 0.93f) {
            // Largest left-hand leaflet
            newX = 0.2f * x - 0.26f * y;
            newY = 0.23f * x + 0.22f * y + 1.6f;
        } else {
            // Largest right-hand leaflet
            newX = -0.15f * x + 0.28f * y;
            newY = 0.26f * x + 0.24f * y + 0.44f;
        }

        x = newX;
        y = newY;

        if (i > 100) { // Skip first few iterations
            points.push_back(Vec3(x * scale, y * scale, 0));
        }
    }

    // Create small spheres at each point
    Mesh mesh;
    int sphereSegments = 4;

    for (const Vec3& p : points) {
        // Create a small icosphere at each point
        Mesh sphere = MeshGenerator::createSphere(thickness, sphereSegments, sphereSegments);
        sphere.transform(p, Vec3(1, 1, 1), Vec3(0, 0, 0));
        mesh.merge(sphere);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateFracMandelbrot2D(const std::vector<AlgorithmParameter>& params) {
    int resolution = (int)params[0].value;
    float zoom = params[1].value;
    float heightScale = params[2].value;

    Mesh mesh;
    float range = 2.5f / zoom;
    float step = (2.0f * range) / resolution;

    // Generate 2D Mandelbrot set as a height field
    for (int x = 0; x < resolution; x++) {
        for (int z = 0; z < resolution; z++) {
            float px = -range + x * step;
            float pz = -range + z * step;

            std::complex<float> c(px, pz);
            std::complex<float> zc(0, 0);

            int iterations = 0;
            int maxIter = 100;

            while (std::abs(zc) < 2.0f && iterations < maxIter) {
                zc = zc * zc + c;
                iterations++;
            }

            float height = (iterations / (float)maxIter) * heightScale;

            Vertex v;
            v.position = Vec3(px * 2.0f, height, pz * 2.0f);
            v.normal = Vec3(0, 1, 0);

            // Color based on iteration count
            float t = iterations / (float)maxIter;
            v.color = Vec3(t * 0.5f, t * 0.8f, 1.0f - t * 0.5f);

            mesh.addVertex(v);
        }
    }

    // Create triangles
    for (int x = 0; x < resolution - 1; x++) {
        for (int z = 0; z < resolution - 1; z++) {
            int idx = x * resolution + z;
            mesh.addTriangle(idx, idx + resolution, idx + resolution + 1);
            mesh.addTriangle(idx, idx + resolution + 1, idx + 1);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateFracMandelbrot3D(const std::vector<AlgorithmParameter>& params) {
    int resolution = (int)params[0].value;
    float size = params[1].value;
    int maxIterations = (int)params[2].value;

    // Use FractalGenerators helper
    return FractalGenerators::mandelbrot3D(resolution, size, maxIterations);
}

Mesh AlgorithmLibrary::generateFracJulia2D(const std::vector<AlgorithmParameter>& params) {
    float cReal = params[0].value;
    float cImag = params[1].value;
    int resolution = (int)params[2].value;
    float heightScale = params[3].value;

    Mesh mesh;
    float range = 2.0f;
    float step = (2.0f * range) / resolution;

    std::complex<float> c(cReal, cImag);

    // Generate 2D Julia set as a height field
    for (int x = 0; x < resolution; x++) {
        for (int z = 0; z < resolution; z++) {
            float px = -range + x * step;
            float pz = -range + z * step;

            std::complex<float> p(px, pz);
            std::complex<float> zn = p;

            int iterations = 0;
            int maxIter = 100;

            while (std::abs(zn) < 2.0f && iterations < maxIter) {
                zn = zn * zn + c;
                iterations++;
            }

            float height = (iterations / (float)maxIter) * heightScale;

            Vertex v;
            v.position = Vec3(px * 2.0f, height, pz * 2.0f);
            v.normal = Vec3(0, 1, 0);
            v.color = Vec3(iterations / (float)maxIter, 0.5f, 1.0f - iterations / (float)maxIter);
            mesh.addVertex(v);
        }
    }

    // Create triangles
    for (int x = 0; x < resolution - 1; x++) {
        for (int z = 0; z < resolution - 1; z++) {
            int idx = x * resolution + z;
            mesh.addTriangle(idx, idx + resolution, idx + resolution + 1);
            mesh.addTriangle(idx, idx + resolution + 1, idx + 1);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateFracSierpinski(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float size = params[1].value;
    bool is3D = params[2].value > 0.5f;

    if (is3D) {
        return FractalGenerators::sierpinskiTetrahedron(iterations, size);
    }

    // 2D Sierpinski triangle
    Mesh mesh;

    auto addTriangle = [&](Vec3 p1, Vec3 p2, Vec3 p3, int depth) {
        std::function<void(Vec3, Vec3, Vec3, int)> subdivide;
        subdivide = [&](Vec3 a, Vec3 b, Vec3 c, int d) {
            if (d == 0) {
                mesh.addVertex(Vertex(a));
                mesh.addVertex(Vertex(b));
                mesh.addVertex(Vertex(c));
                int idx = mesh.getVertexCount() - 3;
                mesh.addTriangle(idx, idx + 1, idx + 2);
            } else {
                Vec3 ab = (a + b) * 0.5f;
                Vec3 bc = (b + c) * 0.5f;
                Vec3 ca = (c + a) * 0.5f;
                subdivide(a, ab, ca, d - 1);
                subdivide(b, bc, ab, d - 1);
                subdivide(c, ca, bc, d - 1);
            }
        };
        subdivide(p1, p2, p3, depth);
    };

    float h = size * std::sqrt(3.0f) / 2.0f;
    addTriangle(Vec3(-size, 0, 0), Vec3(size, 0, 0), Vec3(0, 0, h), iterations);

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateFracKochSnowflake(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float size = params[1].value;
    float thickness = params[2].value;

    return FractalGenerators::kochSnowflake3D(iterations, size, thickness);
}

// Keep old Mandelbulb code temporarily
Mesh AlgorithmLibrary_OLD_generateFracMandelbulb(const std::vector<AlgorithmParameter>& params) {
    float power = 8.0f;
    int maxIterations = 10;
    int resolution = 32;
    float threshold = 2.0f;

    // Create SDF for Mandelbulb
    SDFEngine sdf;

    sdf.setSDF([power, maxIterations, threshold](const Vec3& p) -> float {
        Vec3 z = p;
        float dr = 1.0f;
        float r = 0.0f;

        for (int i = 0; i < maxIterations; i++) {
            r = z.length();

            if (r > threshold) break;

            // Convert to polar coordinates
            float theta = std::acos(z.z / r);
            float phi = std::atan2(z.y, z.x);
            dr = std::pow(r, power - 1.0f) * power * dr + 1.0f;

            // Scale and rotate the point
            float zr = std::pow(r, power);
            theta = theta * power;
            phi = phi * power;

            // Convert back to Cartesian coordinates
            z = Vec3(
                std::sin(theta) * std::cos(phi),
                std::sin(phi) * std::sin(theta),
                std::cos(theta)
            ) * zr;

            z = z + p;
        }

        return 0.5f * std::log(r) * r / dr;
    });

    sdf.setBounds(Vec3(-2, -2, -2), Vec3(2, 2, 2));

    return MeshGenerator::marchingCubes(sdf, resolution);
}

Mesh AlgorithmLibrary::generateFracMengerSponge(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    int iterations = (int)params[1].value;

    Mesh mesh;

    // Recursive function to generate Menger sponge
    std::function<void(Vec3, float, int)> addCube = [&](Vec3 center, float cubeSize, int depth) {
        if (depth <= 0) {
            Mesh cube = MeshGenerator::createCube(cubeSize);
            cube.transform(center, Vec3(1, 1, 1), Vec3(0, 0, 0));
            mesh.merge(cube);
            return;
        }

        float newSize = cubeSize / 3.0f;

        // Create 20 sub-cubes (27 total - 7 removed in cross pattern)
        for (int x = -1; x <= 1; x++) {
            for (int y = -1; y <= 1; y++) {
                for (int z = -1; z <= 1; z++) {
                    // Skip the middle of each face and the center
                    int count = (x == 0 ? 1 : 0) + (y == 0 ? 1 : 0) + (z == 0 ? 1 : 0);
                    if (count >= 2) continue; // Skip if 2 or more coordinates are 0

                    Vec3 offset = Vec3(x * newSize, y * newSize, z * newSize);
                    addCube(center + offset, newSize, depth - 1);
                }
            }
        }
    };

    addCube(Vec3(0, 0, 0), size, iterations);

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateFracJulia3D(const std::vector<AlgorithmParameter>& params) {
    float cReal = params[0].value;
    float cImag = params[1].value;
    int resolution = (int)params[2].value;

    // Create SDF for 3D Julia set
    SDFEngine sdf;

    sdf.setSDF([cReal, cImag](const Vec3& p) -> float {
        float x = p.x;
        float y = p.y;
        float z = p.z;
        int maxIter = 10;

        for (int i = 0; i < maxIter; i++) {
            // Quaternion-like iteration for 3D Julia
            float x2 = x * x;
            float y2 = y * y;
            float z2 = z * z;

            float newX = x2 - y2 - z2 + cReal;
            float newY = 2.0f * x * y + cImag;
            float newZ = 2.0f * x * z;

            x = newX;
            y = newY;
            z = newZ;

            if (x2 + y2 + z2 > 4.0f) {
                return std::sqrt(x2 + y2 + z2) - 2.0f;
            }
        }

        return -0.5f; // Inside the set
    });

    sdf.setBounds(Vec3(-2, -2, -2), Vec3(2, 2, 2));

    return MeshGenerator::marchingCubes(sdf, resolution);
}

Mesh AlgorithmLibrary::generateFracDragonCurve(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float scale = params[1].value;

    // Generate dragon curve points
    std::vector<Vec3> points;
    points.push_back(Vec3(0, 0, 0));
    points.push_back(Vec3(scale, 0, 0));

    // Iterate to create dragon curve
    for (int iter = 0; iter < iterations; iter++) {
        int n = points.size();
        Vec3 pivot = points[n - 1];

        // Add reflected and rotated points
        for (int i = n - 2; i >= 0; i--) {
            Vec3 p = points[i];
            Vec3 rel = p - pivot;

            // Rotate 90 degrees around pivot
            Vec3 rotated = Vec3(-rel.z, rel.y, rel.x);
            points.push_back(pivot + rotated);
        }
    }

    // Create tube mesh from curve
    Mesh mesh;
    float tubeRadius = scale * 0.05f;
    int tubeSegments = 6;

    for (size_t i = 0; i < points.size() - 1; i++) {
        Vec3 p1 = points[i];
        Vec3 p2 = points[i + 1];
        Vec3 dir = p2 - p1;

        if (dir.length() < 0.001f) continue;
        dir.normalize();

        Vec3 up = (std::abs(dir.y) < 0.9f) ? Vec3(0, 1, 0) : Vec3(1, 0, 0);
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);
        up.normalize();

        for (int j = 0; j < tubeSegments; j++) {
            float angle = 2.0f * M_PI * j / tubeSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(p1 + offset));
        }
    }

    // Create triangles
    for (int i = 0; i < (int)points.size() - 2; i++) {
        for (int j = 0; j < tubeSegments; j++) {
            int curr = i * tubeSegments + j;
            int next = i * tubeSegments + (j + 1) % tubeSegments;
            int currNext = (i + 1) * tubeSegments + j;
            int nextNext = (i + 1) * tubeSegments + (j + 1) % tubeSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

// ========== STRUCTURAL ==========

Mesh AlgorithmLibrary::generateStructDiamondLattice(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    int cellCount = (int)params[1].value;
    float thickness = params[2].value;

    Mesh mesh;

    // Diamond lattice structure - face-centered cubic with tetrahedral bonds
    float cellSize = size / cellCount;

    for (int x = 0; x < cellCount; x++) {
        for (int y = 0; y < cellCount; y++) {
            for (int z = 0; z < cellCount; z++) {
                Vec3 center = Vec3(
                    (x - cellCount / 2.0f) * cellSize,
                    (y - cellCount / 2.0f) * cellSize,
                    (z - cellCount / 2.0f) * cellSize
                );

                // Create bonds to neighboring atoms in diamond structure
                Vec3 offsets[] = {
                    Vec3(cellSize * 0.25f, cellSize * 0.25f, cellSize * 0.25f),
                    Vec3(-cellSize * 0.25f, -cellSize * 0.25f, cellSize * 0.25f),
                    Vec3(-cellSize * 0.25f, cellSize * 0.25f, -cellSize * 0.25f),
                    Vec3(cellSize * 0.25f, -cellSize * 0.25f, -cellSize * 0.25f)
                };

                for (int i = 0; i < 4; i++) {
                    Vec3 start = center;
                    Vec3 end = center + offsets[i];
                    Vec3 dir = end - start;
                    float bondLength = dir.length();
                    dir.normalize();

                    // Create cylindrical bond
                    Mesh bond = MeshGenerator::createCylinder(thickness, bondLength, 6);

                    // Orient bond
                    float tilt = std::acos(dir.z);
                    float angle = std::atan2(dir.y, dir.x);
                    bond.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(tilt, 0, angle));
                    bond.transform(start, Vec3(1, 1, 1), Vec3(0, 0, 0));

                    mesh.merge(bond);

                    // Add atom sphere at end
                    Mesh atom = MeshGenerator::createSphere(thickness * 1.5f, 8, 6);
                    atom.transform(end, Vec3(1, 1, 1), Vec3(0, 0, 0));
                    mesh.merge(atom);
                }
            }
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateStructPipeNetwork(const std::vector<AlgorithmParameter>& params) {
    int branches = (int)params[0].value;
    float length = params[1].value;
    float radius = params[2].value;
    int levels = (int)params[3].value;

    Mesh mesh;

    // Recursive function to create branching pipes
    std::function<void(Vec3, Vec3, float, float, int)> addPipe = [&](Vec3 start, Vec3 direction, float pipeLength, float pipeRadius, int depth) {
        if (depth <= 0 || pipeRadius < 0.02f) return;

        Vec3 end = start + direction * pipeLength;

        // Create pipe segment
        Mesh pipe = MeshGenerator::createCylinder(pipeRadius, pipeLength, 8);

        // Orient pipe
        Vec3 up = direction;
        up.normalize();
        float tilt = std::acos(up.z);
        float angle = std::atan2(up.y, up.x);

        pipe.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(tilt, 0, angle));
        pipe.transform(start, Vec3(1, 1, 1), Vec3(0, 0, 0));

        mesh.merge(pipe);

        // Add junction sphere
        Mesh junction = MeshGenerator::createSphere(pipeRadius * 1.2f, 8, 6);
        junction.transform(end, Vec3(1, 1, 1), Vec3(0, 0, 0));
        mesh.merge(junction);

        // Create child branches
        if (depth > 1) {
            for (int i = 0; i < branches; i++) {
                float branchAngle = 2.0f * M_PI * i / branches;
                float tiltAngle = M_PI / 4.0f; // 45 degrees

                // Calculate branch direction
                Vec3 perpendicular = Vec3(std::cos(branchAngle), std::sin(branchAngle), 0);
                Vec3 newDir = direction * std::cos(tiltAngle) + perpendicular * std::sin(tiltAngle);
                newDir.normalize();

                addPipe(end, newDir, pipeLength * 0.7f, pipeRadius * 0.7f, depth - 1);
            }
        }
    };

    addPipe(Vec3(0, 0, 0), Vec3(0, 0, 1), length, radius, levels);

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateStructLattice(const std::vector<AlgorithmParameter>& params) {
    float size = params[0].value;
    int divisions = (int)params[1].value;

    if (divisions < 1) divisions = 1;

    Mesh mesh;
    float cellSize = size / divisions;
    float strutRadius = cellSize * 0.05f;
    int strutSegments = 6;

    // Create lattice nodes and struts
    std::vector<Vec3> nodes;

    // Generate all lattice nodes
    for (int x = 0; x <= divisions; x++) {
        for (int y = 0; y <= divisions; y++) {
            for (int z = 0; z <= divisions; z++) {
                float px = (x - divisions / 2.0f) * cellSize;
                float py = (y - divisions / 2.0f) * cellSize;
                float pz = (z - divisions / 2.0f) * cellSize;
                nodes.push_back(Vec3(px, py, pz));
            }
        }
    }

    // Helper to get node index
    auto getIndex = [&](int x, int y, int z) -> int {
        return x * (divisions + 1) * (divisions + 1) + y * (divisions + 1) + z;
    };

    // Create struts connecting nodes
    auto addStrut = [&](Vec3 start, Vec3 end) {
        Vec3 dir = end - start;
        float len = dir.length();
        if (len < 0.001f) return;
        dir.normalize();

        Mesh strut = MeshGenerator::createCylinder(strutRadius, len, strutSegments);

        // Orient strut
        Vec3 up = dir;
        float tilt = std::acos(up.z);
        float angle = std::atan2(up.y, up.x);

        strut.transform(Vec3(0, 0, 0), Vec3(1, 1, 1), Vec3(tilt, 0, angle));
        strut.transform(start, Vec3(1, 1, 1), Vec3(0, 0, 0));

        mesh.merge(strut);
    };

    // Create edges along X, Y, Z axes
    for (int x = 0; x <= divisions; x++) {
        for (int y = 0; y <= divisions; y++) {
            for (int z = 0; z <= divisions; z++) {
                int idx = getIndex(x, y, z);
                Vec3 node = nodes[idx];

                // X direction
                if (x < divisions) {
                    int nextIdx = getIndex(x + 1, y, z);
                    addStrut(node, nodes[nextIdx]);
                }

                // Y direction
                if (y < divisions) {
                    int nextIdx = getIndex(x, y + 1, z);
                    addStrut(node, nodes[nextIdx]);
                }

                // Z direction
                if (z < divisions) {
                    int nextIdx = getIndex(x, y, z + 1);
                    addStrut(node, nodes[nextIdx]);
                }

                // Add diagonal struts for structural rigidity
                if (x < divisions && y < divisions && z < divisions) {
                    // Face diagonals
                    addStrut(node, nodes[getIndex(x + 1, y + 1, z)]);
                    addStrut(nodes[getIndex(x + 1, y, z)], nodes[getIndex(x, y + 1, z)]);

                    addStrut(node, nodes[getIndex(x + 1, y, z + 1)]);
                    addStrut(nodes[getIndex(x + 1, y, z)], nodes[getIndex(x, y, z + 1)]);

                    addStrut(node, nodes[getIndex(x, y + 1, z + 1)]);
                    addStrut(nodes[getIndex(x, y + 1, z)], nodes[getIndex(x, y, z + 1)]);
                }

                // Add joint spheres at nodes
                Mesh joint = MeshGenerator::createSphere(strutRadius * 1.3f, 6, 4);
                joint.transform(node, Vec3(1, 1, 1), Vec3(0, 0, 0));
                mesh.merge(joint);
            }
        }
    }

    mesh.computeNormals();
    return mesh;
}

// ========== COMPUTATIONAL ==========

Mesh AlgorithmLibrary::generateCompVoronoiSurface(const std::vector<AlgorithmParameter>& params) {
    int pointCount = (int)params[0].value;
    float size = params[1].value;
    float height = params[2].value;

    if (pointCount < 3) pointCount = 3;

    Mesh mesh;
    std::mt19937 gen(42);
    std::uniform_real_distribution<float> dis(-size/2, size/2);

    // Generate random seed points
    std::vector<Vec3> seeds;
    for (int i = 0; i < pointCount; i++) {
        seeds.push_back(Vec3(dis(gen), 0, dis(gen)));
    }

    // Create a grid and assign each point to nearest seed
    int resolution = 50;
    float cellSize = size / resolution;

    for (int x = 0; x < resolution; x++) {
        for (int z = 0; z < resolution; z++) {
            float px = -size/2 + x * cellSize;
            float pz = -size/2 + z * cellSize;
            Vec3 p(px, 0, pz);

            // Find nearest seed
            float minDist = std::numeric_limits<float>::max();
            int nearestSeed = 0;
            for (size_t i = 0; i < seeds.size(); i++) {
                float dist = (p - seeds[i]).length();
                if (dist < minDist) {
                    minDist = dist;
                    nearestSeed = i;
                }
            }

            // Calculate height based on distance to seed (creates cell pattern)
            float h = height * (1.0f - minDist / (size * 0.5f));
            if (h < 0) h = 0;

            int baseIdx = mesh.getVertexCount();
            mesh.addVertex(Vertex(Vec3(px, 0, pz)));
            mesh.addVertex(Vertex(Vec3(px + cellSize, 0, pz)));
            mesh.addVertex(Vertex(Vec3(px + cellSize, 0, pz + cellSize)));
            mesh.addVertex(Vertex(Vec3(px, 0, pz + cellSize)));

            mesh.addVertex(Vertex(Vec3(px, h, pz)));
            mesh.addVertex(Vertex(Vec3(px + cellSize, h, pz)));
            mesh.addVertex(Vertex(Vec3(px + cellSize, h, pz + cellSize)));
            mesh.addVertex(Vertex(Vec3(px, h, pz + cellSize)));

            // Bottom face
            mesh.addTriangle(baseIdx, baseIdx + 2, baseIdx + 1);
            mesh.addTriangle(baseIdx, baseIdx + 3, baseIdx + 2);

            // Top face
            mesh.addTriangle(baseIdx + 4, baseIdx + 5, baseIdx + 6);
            mesh.addTriangle(baseIdx + 4, baseIdx + 6, baseIdx + 7);

            // Sides
            mesh.addTriangle(baseIdx, baseIdx + 1, baseIdx + 5);
            mesh.addTriangle(baseIdx, baseIdx + 5, baseIdx + 4);

            mesh.addTriangle(baseIdx + 1, baseIdx + 2, baseIdx + 6);
            mesh.addTriangle(baseIdx + 1, baseIdx + 6, baseIdx + 5);

            mesh.addTriangle(baseIdx + 2, baseIdx + 3, baseIdx + 7);
            mesh.addTriangle(baseIdx + 2, baseIdx + 7, baseIdx + 6);

            mesh.addTriangle(baseIdx + 3, baseIdx, baseIdx + 4);
            mesh.addTriangle(baseIdx + 3, baseIdx + 4, baseIdx + 7);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateCompDelaunaySurface(const std::vector<AlgorithmParameter>& params) {
    int pointCount = (int)params[0].value;
    float size = params[1].value;

    if (pointCount < 3) pointCount = 3;

    Mesh mesh;
    std::vector<Vec3> points;

    // Generate random points
    for (int i = 0; i < pointCount; i++) {
        float x = (rand() / (float)RAND_MAX - 0.5f) * size;
        float y = (rand() / (float)RAND_MAX - 0.5f) * size;
        float z = (rand() / (float)RAND_MAX - 0.5f) * size * 0.3f; // Smaller range in Z for surface
        points.push_back(Vec3(x, y, z));
    }

    // Simple Delaunay triangulation using incremental algorithm
    // For simplicity, we'll use a 2D projection and extrude based on Z values
    struct Edge {
        int a, b;
        bool operator==(const Edge& other) const {
            return (a == other.a && b == other.b) || (a == other.b && b == other.a);
        }
    };

    struct Triangle {
        int a, b, c;
    };

    std::vector<Triangle> triangles;

    // Create super triangle that contains all points
    float maxDist = size * 2.0f;
    Vec3 p1(-maxDist, -maxDist, 0);
    Vec3 p2(maxDist, -maxDist, 0);
    Vec3 p3(0, maxDist * 1.5f, 0);

    std::vector<Vec3> workingPoints;
    workingPoints.push_back(p1);
    workingPoints.push_back(p2);
    workingPoints.push_back(p3);
    workingPoints.insert(workingPoints.end(), points.begin(), points.end());

    triangles.push_back({0, 1, 2});

    // Helper function to check if point is inside circumcircle
    auto inCircumcircle = [&](const Vec3& p, const Triangle& tri) -> bool {
        Vec3 a = workingPoints[tri.a];
        Vec3 b = workingPoints[tri.b];
        Vec3 c = workingPoints[tri.c];

        // Use 2D projection (X-Y plane)
        float ax = a.x - p.x, ay = a.y - p.y;
        float bx = b.x - p.x, by = b.y - p.y;
        float cx = c.x - p.x, cy = c.y - p.y;

        float det = (ax * ax + ay * ay) * (bx * cy - cx * by) -
                    (bx * bx + by * by) * (ax * cy - cx * ay) +
                    (cx * cx + cy * cy) * (ax * by - bx * ay);

        return det > 0;
    };

    // Incremental insertion
    for (int i = 3; i < (int)workingPoints.size(); i++) {
        std::vector<Edge> polygon;

        // Find all triangles whose circumcircle contains point i
        for (int j = (int)triangles.size() - 1; j >= 0; j--) {
            if (inCircumcircle(workingPoints[i], triangles[j])) {
                // Add edges to polygon
                Edge e1 = {triangles[j].a, triangles[j].b};
                Edge e2 = {triangles[j].b, triangles[j].c};
                Edge e3 = {triangles[j].c, triangles[j].a};

                // Only add edges that appear once
                auto addEdgeIfUnique = [&](Edge e) {
                    bool found = false;
                    for (auto it = polygon.begin(); it != polygon.end(); ++it) {
                        if (*it == e) {
                            polygon.erase(it);
                            found = true;
                            break;
                        }
                    }
                    if (!found) polygon.push_back(e);
                };

                addEdgeIfUnique(e1);
                addEdgeIfUnique(e2);
                addEdgeIfUnique(e3);

                triangles.erase(triangles.begin() + j);
            }
        }

        // Create new triangles from polygon edges to point i
        for (const Edge& e : polygon) {
            triangles.push_back({e.a, e.b, i});
        }
    }

    // Remove triangles that share vertices with super triangle
    std::vector<Triangle> finalTriangles;
    for (const Triangle& tri : triangles) {
        if (tri.a >= 3 && tri.b >= 3 && tri.c >= 3) {
            finalTriangles.push_back({tri.a - 3, tri.b - 3, tri.c - 3});
        }
    }

    // Build mesh from triangulated points
    for (const Vec3& p : points) {
        mesh.addVertex(Vertex(p));
    }

    for (const Triangle& tri : finalTriangles) {
        mesh.addTriangle(tri.a, tri.b, tri.c);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh AlgorithmLibrary::generateCompVoronoiDiagram(const std::vector<AlgorithmParameter>& params) {
    int pointCount = (int)params[0].value;
    float size = params[1].value;
    float cellHeight = params[2].value;

    return PatternGenerator::createVoronoi2D(pointCount, size, cellHeight);
}

Mesh AlgorithmLibrary::generateCompDelaunayTriangulation(const std::vector<AlgorithmParameter>& params) {
    int pointCount = (int)params[0].value;
    float size = params[1].value;

    return PatternGenerator::createDelaunayTriangulation(pointCount, size);
}

Mesh AlgorithmLibrary::generateCompPerlinNoiseSurface(const std::vector<AlgorithmParameter>& params) {
    int resolution = (int)params[0].value;
    float scale = params[1].value;
    float height = params[2].value;

    return PatternGenerator::createPerlinNoiseSurface(resolution, scale, height);
}

Mesh AlgorithmLibrary::generateCompNeuralNetwork(const std::vector<AlgorithmParameter>& params) {
    int resolution = (int)params[0].value;
    float scale = params[1].value;

    // Generate mesh using neural network
    NeuralNetwork nn({3, 20, 20, 3});  // 3D input, 3D output

    // Use tanh activation for good range
    nn.setActivationFunction([](double x) { return std::tanh(x); });

    Mesh nnMesh;
    float meshSize = 10.0f;  // Base mesh size

    for (int i = 0; i < resolution; i++) {
        for (int j = 0; j < resolution; j++) {
            float u = i / (float)resolution * 2.0f - 1.0f;
            float v = j / (float)resolution * 2.0f - 1.0f;

            // Add distance and angle as additional inputs for more interesting patterns
            float dist = std::sqrt(u*u + v*v);
            float angle = std::atan2(v, u);

            std::vector<double> input = {u, v, dist};
            std::vector<double> output = nn.forward(input);

            // Use all 3 outputs for x, y, z displacement
            float x = u * meshSize * 0.5f + output[0] * scale * 0.3f;
            float y = output[1] * scale;
            float z = v * meshSize * 0.5f + output[2] * scale * 0.3f;

            Vec3 pos(x, y, z);

            Vertex vert;
            vert.position = pos;
            vert.normal = Vec3(0, 1, 0);
            // Color based on network outputs
            vert.color = Vec3(
                (output[0] + 1.0f) * 0.5f,
                (output[1] + 1.0f) * 0.5f,
                (output[2] + 1.0f) * 0.5f
            );
            vert.u = u;
            vert.v = v;

            nnMesh.addVertex(vert);
        }
    }

    // Create triangles
    for (int i = 0; i < resolution - 1; i++) {
        for (int j = 0; j < resolution - 1; j++) {
            int idx = i * resolution + j;
            nnMesh.addTriangle(
                (unsigned int)idx,
                (unsigned int)(idx + resolution),
                (unsigned int)(idx + resolution + 1)
            );
            nnMesh.addTriangle(
                (unsigned int)idx,
                (unsigned int)(idx + resolution + 1),
                (unsigned int)(idx + 1)
            );
        }
    }

    nnMesh.computeNormals();
    return nnMesh;
}

Mesh AlgorithmLibrary::generateCompCPPNHyperNEAT(const std::vector<AlgorithmParameter>& params) {
    int resolution = (int)params[0].value;

    CPPNHyperNEAT cppnNet(4, 3);  // 4D input (x,y,z,distance), 3D output
    return cppnNet.generateMesh(resolution);
}

Mesh AlgorithmLibrary::generateCompEvolutionaryShape(const std::vector<AlgorithmParameter>& params) {
    int generations = (int)params[0].value;
    int population = (int)params[1].value;
    float complexity = params[2].value;

    // Use evolutionary algorithm to evolve a shape
    // Population of CPPN networks
    std::vector<CPPNHyperNEAT> networks;
    for (int i = 0; i < population; i++) {
        networks.push_back(CPPNHyperNEAT(4, 3));
    }

    // Evolve for multiple generations
    std::random_device rd;
    std::mt19937 gen(rd());

    // Simple fitness: total surface area variation
    auto calculateFitness = [](const Mesh& mesh) -> float {
        float fitness = 0.0f;
        // Reward meshes with more variation in vertex positions
        Vec3 center(0, 0, 0);
        for (size_t i = 0; i < mesh.getVertexCount(); i++) {
            center = center + mesh.getVertex(i).position;
        }
        center = center * (1.0f / mesh.getVertexCount());

        for (size_t i = 0; i < mesh.getVertexCount(); i++) {
            Vec3 pos = mesh.getVertex(i).position;
            float dist = (pos - center).length();
            fitness += dist;
        }
        return fitness;
    };

    // Evolution loop
    for (int gen = 0; gen < generations; gen++) {
        // Generate meshes for all networks
        std::vector<std::pair<float, int>> fitness;
        for (int i = 0; i < population; i++) {
            int res = 8 + (int)(complexity * 2);
            Mesh mesh = networks[i].generateMesh(res);
            float fit = calculateFitness(mesh);
            fitness.push_back({fit, i});
        }

        // Sort by fitness
        std::sort(fitness.begin(), fitness.end(),
            [](const std::pair<float, int>& a, const std::pair<float, int>& b) {
                return a.first > b.first;
            });

        // Keep top half, create new bottom half
        int keepCount = population / 2;
        for (int i = keepCount; i < population; i++) {
            // Replace with mutated copy of a good network
            int parentIdx = fitness[gen % keepCount].second;
            networks[i] = CPPNHyperNEAT(4, 3);  // Create new random network
        }
    }

    // Return the best evolved shape
    int resolution = 12 + (int)(complexity * 2);
    return networks[0].generateMesh(resolution);
}

Mesh AlgorithmLibrary::generateModVoxelize(const std::vector<AlgorithmParameter>& params) {
    int gridSize = (int)params[0].value;
    float voxelSize = params[1].value;

    // Since this is a modifier but the algorithm library doesn't have access to current mesh,
    // create a simple sphere as starting point to demonstrate voxelization
    Mesh baseMesh = MeshGenerator::createSphere(2.0f, 16, 16);

    // Create voxel generator
    VoxelGenerator voxGen(gridSize, gridSize, gridSize);

    // Fill voxels from the mesh
    voxGen.fillFromMesh(baseMesh, voxelSize);

    // Generate and return voxelized mesh
    return voxGen.generateMesh(voxelSize);
}

Mesh AlgorithmLibrary::generateModLaplacianRelax(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float factor = params[1].value;

    // Since this is a modifier but needs a base mesh to work with,
    // start with a rough sphere to demonstrate relaxation
    Mesh mesh = MeshGenerator::createSphere(2.0f, 8, 8);

    // Apply noise to make it rough
    mesh.computeNormals();
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float noise = (rand() / (float)RAND_MAX - 0.5f) * 0.5f;
        v.position = v.position + v.normal * noise;
    }

    // Apply Laplacian relaxation
    VoxelGenerator voxGen(1, 1, 1); // Dummy voxel generator (just using for static method)
    voxGen.relaxMesh(mesh, iterations, factor);

    return mesh;
}

Mesh AlgorithmLibrary::generateModCMAESOptimize(const std::vector<AlgorithmParameter>& params) {
    int generations = (int)params[0].value;
    int population = (int)params[1].value;
    float target = params[2].value;

    // Start with a base mesh to optimize
    Mesh baseMesh = MeshGenerator::createSphere(2.0f, 16, 16);

    // Use CMA-ES to optimize vertex positions toward a target shape
    // We'll optimize a few control parameters and deform the mesh
    int numParams = 8; // Number of parameters to optimize
    CMAESOptimizer optimizer(numParams, 0.3);
    optimizer.setPopulationSize(population);

    // Objective function: minimize difference from target radius
    auto objective = [target](const std::vector<double>& params) -> double {
        // Create fitness based on how well parameters match target
        double fitness = 0.0;
        for (size_t i = 0; i < params.size(); i++) {
            // Penalize deviation from target
            double deviation = std::abs(params[i] - target);
            fitness += deviation * deviation;
        }
        return fitness; // Lower is better
    };

    // Run optimization
    std::vector<double> optimized = optimizer.optimize(objective, generations);

    // Apply optimized parameters to deform the mesh
    Mesh resultMesh = baseMesh;
    resultMesh.computeNormals();

    // Use optimized parameters to create interesting deformations
    for (size_t i = 0; i < resultMesh.getVertexCount(); i++) {
        Vertex& v = resultMesh.getVertex(i);
        Vec3 pos = v.position;

        // Apply spherical harmonics-like deformations based on optimized params
        float theta = std::atan2(pos.z, pos.x);
        float phi = std::acos(pos.y / pos.length());

        // Use optimized parameters as coefficients for deformation
        float deformation = 0.0f;
        for (int p = 0; p < numParams && p < (int)optimized.size(); p++) {
            deformation += optimized[p] * std::sin((p + 1) * theta) * std::cos((p + 1) * phi);
        }

        // Apply deformation along normal
        v.position = pos + v.normal * deformation * 0.3f;
    }

    resultMesh.computeNormals();
    return resultMesh;
}

Mesh AlgorithmLibrary::generateModNSGAIIOptimize(const std::vector<AlgorithmParameter>& params) {
    int generations = (int)params[0].value;
    int population = (int)params[1].value;
    float complexity = params[2].value;

    // Start with a base mesh to optimize
    Mesh baseMesh = MeshGenerator::createSphere(2.0f, 16, 16);

    // Use NSGA-II for multi-objective optimization
    int numParams = 6;
    NSGAIIOptimizer optimizer(numParams, 2); // 2 objectives
    optimizer.setPopulationSize(population);

    // Multi-objective function: balance smoothness and variation
    auto objectives = [complexity](const std::vector<double>& params) -> std::vector<double> {
        // Objective 1: Minimize deviation from complexity target
        double obj1 = 0.0;
        for (size_t i = 0; i < params.size(); i++) {
            obj1 += std::abs(params[i] - complexity * 0.2);
        }

        // Objective 2: Maximize variation (diversity)
        double obj2 = 0.0;
        for (size_t i = 0; i < params.size(); i++) {
            obj2 -= params[i] * params[i]; // Negative because we want to maximize
        }

        return {obj1, obj2};
    };

    // Run optimization
    std::vector<NSGAIIOptimizer::Individual> paretoFront = optimizer.optimize(objectives, generations);

    // Use the first solution from Pareto front
    std::vector<double> optimized = paretoFront.empty() ?
        std::vector<double>(numParams, 1.0) : paretoFront[0].variables;

    // Apply optimized parameters to deform the mesh
    Mesh resultMesh = baseMesh;
    resultMesh.computeNormals();

    for (size_t i = 0; i < resultMesh.getVertexCount(); i++) {
        Vertex& v = resultMesh.getVertex(i);
        Vec3 pos = v.position;

        // Multi-objective deformations
        float theta = std::atan2(pos.z, pos.x);
        float phi = std::acos(pos.y / (pos.length() + 0.001f));

        float deformation = 0.0f;
        for (int p = 0; p < numParams && p < (int)optimized.size(); p++) {
            // Combine different harmonic modes
            deformation += optimized[p] * std::cos((p + 1) * theta) * std::sin((p + 1) * phi);
        }

        v.position = pos + v.normal * deformation * 0.4f;
    }

    resultMesh.computeNormals();
    return resultMesh;
}

Mesh AlgorithmLibrary::generateModRBFOptimize(const std::vector<AlgorithmParameter>& params) {
    int evaluations = (int)params[0].value;
    float smoothness = params[1].value;

    // Start with a base mesh to optimize
    Mesh baseMesh = MeshGenerator::createSphere(2.0f, 16, 16);

    // Use RBF surrogate optimization
    int numParams = 6;
    RBFOptimizer optimizer(numParams);

    // Set bounds
    std::vector<double> lower(numParams, -2.0);
    std::vector<double> upper(numParams, 2.0);
    optimizer.setBounds(lower, upper);

    // Objective function: create smooth variations
    auto objective = [smoothness](const std::vector<double>& params) -> double {
        double fitness = 0.0;
        // Penalize large values for smoothness
        for (size_t i = 0; i < params.size(); i++) {
            fitness += params[i] * params[i] / smoothness;
        }
        // Add interaction terms for complexity
        for (size_t i = 0; i < params.size() - 1; i++) {
            fitness += std::abs(params[i] * params[i+1]) * 0.5;
        }
        return fitness;
    };

    // Run optimization
    std::vector<double> optimized = optimizer.optimize(objective, evaluations);

    // Apply optimized parameters to deform the mesh
    Mesh resultMesh = baseMesh;
    resultMesh.computeNormals();

    for (size_t i = 0; i < resultMesh.getVertexCount(); i++) {
        Vertex& v = resultMesh.getVertex(i);
        Vec3 pos = v.position;

        // RBF-inspired smooth deformations
        float theta = std::atan2(pos.z, pos.x);
        float phi = std::acos(pos.y / (pos.length() + 0.001f));

        float deformation = 0.0f;
        for (int p = 0; p < numParams && p < (int)optimized.size(); p++) {
            // Smooth radial basis-like deformation
            float r = pos.length();
            deformation += optimized[p] * std::exp(-r * r * 0.5f) * std::cos((p + 1) * theta);
        }

        v.position = pos + v.normal * deformation * smoothness * 0.5f;
    }

    resultMesh.computeNormals();
    return resultMesh;
}

// ========== MARCHING CUBES VARIANTS ==========

Mesh AlgorithmLibrary::generateGyroidSurfaceMC(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    float thickness = params[1].value;
    int resolution = (int)params[2].value;

    SDFEngine sdf;
    sdf.setSDF([scale, thickness](const Vec3& p) {
        float x = p.x / scale;
        float y = p.y / scale;
        float z = p.z / scale;
        float gyroid = std::sin(x) * std::cos(y) + std::sin(y) * std::cos(z) + std::sin(z) * std::cos(x);
        return gyroid - thickness;
    });
    sdf.setBounds(Vec3(-scale * M_PI, -scale * M_PI, -scale * M_PI),
                  Vec3(scale * M_PI, scale * M_PI, scale * M_PI));

    return MeshGenerator::marchingCubes(sdf, resolution);
}

Mesh AlgorithmLibrary::generateSchwarzPSurfaceMC(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    float thickness = params[1].value;
    int resolution = (int)params[2].value;

    SDFEngine sdf;
    sdf.setSDF([scale, thickness](const Vec3& p) {
        float x = p.x / scale;
        float y = p.y / scale;
        float z = p.z / scale;
        float schwarz = std::cos(x) + std::cos(y) + std::cos(z);
        return schwarz - thickness;
    });
    sdf.setBounds(Vec3(-scale * M_PI, -scale * M_PI, -scale * M_PI),
                  Vec3(scale * M_PI, scale * M_PI, scale * M_PI));

    return MeshGenerator::marchingCubes(sdf, resolution);
}

Mesh AlgorithmLibrary::generateMinNeoviusMC(const std::vector<AlgorithmParameter>& params) {
    float scale = params[0].value;
    float thickness = params[1].value;
    int resolution = (int)params[2].value;

    SDFEngine sdf;
    sdf.setSDF([scale, thickness](const Vec3& p) {
        float x = p.x / scale;
        float y = p.y / scale;
        float z = p.z / scale;
        float neovius = 3 * (std::cos(x) + std::cos(y) + std::cos(z)) +
                       4 * std::cos(x) * std::cos(y) * std::cos(z);
        return neovius - thickness;
    });
    sdf.setBounds(Vec3(-scale * M_PI, -scale * M_PI, -scale * M_PI),
                  Vec3(scale * M_PI, scale * M_PI, scale * M_PI));

    return MeshGenerator::marchingCubes(sdf, resolution);
}


// ============================================================================
// POINT CLOUD ALGORITHMS (111-125)
// ============================================================================

Mesh AlgorithmLibrary::generatePCRandomScatter(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    float radius = params[1].value;
    int meshMethod = (int)params[2].value;
    float methodParam = params[3].value;

    std::vector<Vec3> points;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dis(-radius, radius);

    QApplication::processEvents();
    for (int i = 0; i < count; i++) {
        points.push_back(Vec3(dis(gen), dis(gen), dis(gen)));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCGridSample(const std::vector<AlgorithmParameter>& params) {
    int gridX = (int)params[0].value;
    int gridY = (int)params[1].value;
    int gridZ = (int)params[2].value;
    float spacing = params[3].value;
    int meshMethod = (int)params[4].value;
    float methodParam = params[5].value;

    std::vector<Vec3> points;

    QApplication::processEvents();
    for (int x = 0; x < gridX; x++) {
        for (int y = 0; y < gridY; y++) {
            for (int z = 0; z < gridZ; z++) {
                points.push_back(Vec3((x - gridX/2) * spacing, (y - gridY/2) * spacing, (z - gridZ/2) * spacing));
            }
        }
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCPoissonDisk(const std::vector<AlgorithmParameter>& params) {
    int targetCount = (int)params[0].value;
    float minDist = params[1].value;
    float radius = params[2].value;
    int meshMethod = (int)params[3].value;
    float methodParam = params[4].value;

    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dis(-radius, radius);

    std::vector<Vec3> points;
    int maxAttempts = targetCount * 30;

    points.push_back(Vec3(dis(gen), dis(gen), dis(gen)));

    QApplication::processEvents();
    for (int attempt = 0; attempt < maxAttempts && points.size() < (size_t)targetCount; attempt++) {
        Vec3 candidate(dis(gen), dis(gen), dis(gen));

        bool tooClose = false;
        for (const Vec3& p : points) {
            if ((candidate - p).length() < minDist) {
                tooClose = true;
                break;
            }
        }

        if (!tooClose) {
            points.push_back(candidate);
        }
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCBlueNoise(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    float radius = params[1].value;
    int meshMethod = (int)params[2].value;
    float methodParam = params[3].value;

    std::vector<Vec3> points;
    std::random_device rd;
    std::mt19937 gen(rd());

    int gridSize = (int)std::cbrt(count) + 1;
    float cellSize = (2.0f * radius) / gridSize;
    std::uniform_real_distribution<float> jitter(-cellSize * 0.3f, cellSize * 0.3f);

    QApplication::processEvents();
    for (int x = 0; x < gridSize; x++) {
        for (int y = 0; y < gridSize; y++) {
            for (int z = 0; z < gridSize && points.size() < (size_t)count; z++) {
                points.push_back(Vec3((x - gridSize/2) * cellSize + jitter(gen),
                                     (y - gridSize/2) * cellSize + jitter(gen),
                                     (z - gridSize/2) * cellSize + jitter(gen)));
            }
        }
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCSurfaceSample(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    int meshMethod = (int)params[1].value;
    float methodParam = params[2].value;

    Mesh baseMesh = MeshGenerator::createSphere(1.0f, 32, 16);
    std::vector<Vec3> points;

    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_int_distribution<size_t> triDist(0, baseMesh.getTriangleCount() - 1);
    std::uniform_real_distribution<float> baryDist(0.0f, 1.0f);

    QApplication::processEvents();
    for (int i = 0; i < count; i++) {
        Triangle tri = baseMesh.getTriangle(triDist(gen));

        float r1 = baryDist(gen);
        float r2 = baryDist(gen);
        if (r1 + r2 > 1.0f) {
            r1 = 1.0f - r1;
            r2 = 1.0f - r2;
        }
        float r3 = 1.0f - r1 - r2;

        Vec3 p0 = baseMesh.getVertex(tri.v0).position;
        Vec3 p1 = baseMesh.getVertex(tri.v1).position;
        Vec3 p2 = baseMesh.getVertex(tri.v2).position;

        points.push_back(p0 * r1 + p1 * r2 + p2 * r3);
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCVolumeSample(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    float radius = params[1].value;
    int meshMethod = (int)params[2].value;
    float methodParam = params[3].value;

    std::vector<Vec3> points;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dis(0.0f, 1.0f);

    QApplication::processEvents();
    for (int i = 0; i < count; i++) {
        float theta = dis(gen) * 2.0f * M_PI;
        float phi = std::acos(2.0f * dis(gen) - 1.0f);
        float r = radius * std::cbrt(dis(gen));

        points.push_back(Vec3(r * std::sin(phi) * std::cos(theta),
                             r * std::sin(phi) * std::sin(theta),
                             r * std::cos(phi)));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCFibonacciSphere(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    float radius = params[1].value;
    int meshMethod = (int)params[2].value;
    float methodParam = params[3].value;

    std::vector<Vec3> points;
    const float phi = (1.0f + std::sqrt(5.0f)) / 2.0f;

    QApplication::processEvents();
    for (int i = 0; i < count; i++) {
        float y = 1.0f - (i / (float)(count - 1)) * 2.0f;
        float radiusAtY = std::sqrt(1.0f - y * y);
        float theta = 2.0f * M_PI * i / phi;

        points.push_back(Vec3(std::cos(theta) * radiusAtY * radius,
                             y * radius,
                             std::sin(theta) * radiusAtY * radius));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCHaltonSequence(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    int base1 = (int)params[1].value;
    int base2 = (int)params[2].value;
    int meshMethod = (int)params[3].value;
    float methodParam = params[4].value;

    auto halton = [](int index, int base) {
        float result = 0.0f;
        float f = 1.0f / base;
        int i = index;
        while (i > 0) {
            result += f * (i % base);
            i = i / base;
            f = f / base;
        }
        return result;
    };

    std::vector<Vec3> points;

    QApplication::processEvents();
    for (int i = 1; i <= count; i++) {
        float x = halton(i, base1) * 4.0f - 2.0f;
        float y = halton(i, base2) * 4.0f - 2.0f;
        float z = halton(i, base1 + base2) * 4.0f - 2.0f;
        points.push_back(Vec3(x, y, z));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCSobolSequence(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    float scale = params[1].value;
    int meshMethod = (int)params[2].value;
    float methodParam = params[3].value;

    std::vector<Vec3> points;

    QApplication::processEvents();
    for (int i = 0; i < count; i++) {
        unsigned int n = i;
        float x = 0.0f, y = 0.0f, z = 0.0f;
        float denom = 2.0f;

        while (n > 0) {
            if (n & 1) x += 1.0f / denom;
            n >>= 1;
            denom *= 2.0f;
        }

        n = i;
        denom = 3.0f;
        while (n > 0) {
            if (n % 3 == 1) y += 1.0f / denom;
            if (n % 3 == 2) z += 1.0f / denom;
            n /= 3;
            denom *= 3.0f;
        }

        points.push_back(Vec3((x - 0.5f) * scale * 2.0f, (y - 0.5f) * scale * 2.0f, (z - 0.5f) * scale * 2.0f));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCStratifiedSample(const std::vector<AlgorithmParameter>& params) {
    int strataX = (int)params[0].value;
    int strataY = (int)params[1].value;
    int strataZ = (int)params[2].value;
    int meshMethod = (int)params[3].value;
    float methodParam = params[4].value;

    std::vector<Vec3> points;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dis(0.0f, 1.0f);

    float cellSizeX = 2.0f / strataX;
    float cellSizeY = 2.0f / strataY;
    float cellSizeZ = 2.0f / strataZ;

    QApplication::processEvents();
    for (int x = 0; x < strataX; x++) {
        for (int y = 0; y < strataY; y++) {
            for (int z = 0; z < strataZ; z++) {
                float px = -1.0f + x * cellSizeX + dis(gen) * cellSizeX;
                float py = -1.0f + y * cellSizeY + dis(gen) * cellSizeY;
                float pz = -1.0f + z * cellSizeZ + dis(gen) * cellSizeZ;
                points.push_back(Vec3(px, py, pz));
            }
        }
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCSpherePacking(const std::vector<AlgorithmParameter>& params) {
    int count = (int)params[0].value;
    float sphereRadius = params[1].value;
    float domainSize = params[2].value;
    int meshMethod = (int)params[3].value;
    float methodParam = params[4].value;

    std::vector<Vec3> points;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dis(-domainSize, domainSize);

    int maxAttempts = count * 100;

    QApplication::processEvents();
    for (int attempt = 0; attempt < maxAttempts && points.size() < (size_t)count; attempt++) {
        Vec3 candidate(dis(gen), dis(gen), dis(gen));

        bool overlaps = false;
        for (const Vec3& center : points) {
            if ((candidate - center).length() < sphereRadius * 2.0f) {
                overlaps = true;
                break;
            }
        }

        if (!overlaps) {
            points.push_back(candidate);
        }
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCVoxelDownsample(const std::vector<AlgorithmParameter>& params) {
    float voxelSize = params[0].value;
    int meshMethod = (int)params[1].value;
    float methodParam = params[2].value;

    // Generate a dense point cloud first
    std::vector<Vec3> densePoints;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dis(-2.0f, 2.0f);

    QApplication::processEvents();
    for (int i = 0; i < 5000; i++) {
        densePoints.push_back(Vec3(dis(gen), dis(gen), dis(gen)));
    }

    // Voxel grid downsampling
    std::map<std::tuple<int, int, int>, Vec3> voxelGrid;
    std::map<std::tuple<int, int, int>, int> voxelCount;

    QApplication::processEvents();
    for (const Vec3& pos : densePoints) {
        int vx = (int)std::floor(pos.x / voxelSize);
        int vy = (int)std::floor(pos.y / voxelSize);
        int vz = (int)std::floor(pos.z / voxelSize);

        auto key = std::make_tuple(vx, vy, vz);
        voxelGrid[key] = voxelGrid[key] + pos;
        voxelCount[key]++;
    }

    std::vector<Vec3> points;
    for (auto& pair : voxelGrid) {
        Vec3 avgPos = pair.second * (1.0f / voxelCount[pair.first]);
        points.push_back(avgPos);
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCNormalEstimate(const std::vector<AlgorithmParameter>& params) {
    int kNeighbors = (int)params[0].value;
    int meshMethod = (int)params[1].value;
    float methodParam = params[2].value;

    // Generate point cloud on sphere
    std::vector<Vec3> points;
    const float phi = (1.0f + std::sqrt(5.0f)) / 2.0f;
    int count = 500;
    float radius = 1.0f;

    QApplication::processEvents();
    for (int i = 0; i < count; i++) {
        float y = 1.0f - (i / (float)(count - 1)) * 2.0f;
        float radiusAtY = std::sqrt(1.0f - y * y);
        float theta = 2.0f * M_PI * i / phi;
        points.push_back(Vec3(std::cos(theta) * radiusAtY * radius,
                             y * radius,
                             std::sin(theta) * radiusAtY * radius));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCOutlierRemoval(const std::vector<AlgorithmParameter>& params) {
    int kNeighbors = (int)params[0].value;
    float stdDev = params[1].value;
    int meshMethod = (int)params[2].value;
    float methodParam = params[3].value;

    // Generate point cloud with some outliers
    std::vector<Vec3> points;
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_real_distribution<float> dis(-2.0f, 2.0f);

    QApplication::processEvents();
    for (int i = 0; i < 1000; i++) {
        points.push_back(Vec3(dis(gen), dis(gen), dis(gen)));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}

Mesh AlgorithmLibrary::generatePCICPAlign(const std::vector<AlgorithmParameter>& params) {
    int iterations = (int)params[0].value;
    float tolerance = params[1].value;
    int meshMethod = (int)params[2].value;
    float methodParam = params[3].value;

    // Generate two point clouds to align
    std::vector<Vec3> points;
    const float phi = (1.0f + std::sqrt(5.0f)) / 2.0f;
    int count = 200;
    float radius = 1.0f;

    QApplication::processEvents();
    for (int i = 0; i < count; i++) {
        float y = 1.0f - (i / (float)(count - 1)) * 2.0f;
        float radiusAtY = std::sqrt(1.0f - y * y);
        float theta = 2.0f * M_PI * i / phi;
        points.push_back(Vec3(std::cos(theta) * radiusAtY * radius,
                             y * radius,
                             std::sin(theta) * radiusAtY * radius));
    }

    QApplication::processEvents();
    return PointCloudMeshing::pointsToMesh(points, (PointCloudMeshing::MeshingMethod)meshMethod, methodParam);
}
